;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.netty
  (:require [utilis.io :as io]
            [spectator.log :as log])
  (:import [io.netty.util.concurrent GenericFutureListener]
           [io.netty.channel
            Channel
            ChannelPromise
            ChannelPipeline
            ChannelHandlerContext
            ChannelOutboundHandler
            ChannelInboundHandler
            DefaultChannelPromise
            ChannelHandler]
           [io.netty.util
            ResourceLeakDetector
            ResourceLeakDetector$Level]
           [io.netty.buffer ByteBuf Unpooled]
           [java.nio ByteBuffer]
           [java.nio.charset Charset]
           [java.io File FileInputStream InputStream]
           [java.util.concurrent ExecutorService]
           [io.netty.handler.codec.http2
            Http2FrameStream
            Http2DataFrame
            DefaultHttp2DataFrame]))

(def ^:const array-class (class (clojure.core/byte-array 0)))
(def ^Charset charset (Charset/forName "UTF-8"))

(declare to-byte-buf)

(defn with-promise-listener
  [^ChannelPromise p f]
  (.addListener
   p (reify GenericFutureListener
       (operationComplete [_ _]
         (f)))))

(defn run
  [^ExecutorService exec-service handler & args]
  (.submit exec-service
           (reify Runnable
             (run [_]
               (apply handler args)))))

(defn safe-execute
  ^ChannelPromise [ch ^clojure.lang.IFn f]
  (let [^Channel channel (if (instance? Channel ch)
                           ch
                           (.channel ^ChannelHandlerContext ch))
        event-loop (.eventLoop channel)
        p (DefaultChannelPromise. channel)]
    (if (.inEventLoop event-loop)
      (do (f) (.setSuccess p))
      (.execute event-loop
                (fn []
                  (try
                    (f)
                    (.setSuccess p)
                    (catch Exception e
                      (if (instance? Channel ch)
                        (.fireExceptionCaught (.pipeline ^Channel ch) e)
                        (.fireExceptionCaught ^ChannelHandlerContext ch e))
                      (log/error [::safe-execute :event-loop :error] e)
                      (.setFailure p e))))))
    p))

(defn invoke-write
  (^ChannelPromise [^ChannelPipeline pipeline ^String handler-name msg]
   (invoke-write
    pipeline handler-name msg
    (DefaultChannelPromise. (.channel pipeline))))
  (^ChannelPromise [^ChannelPipeline pipeline ^String handler-name msg p]
   (when-let [writer (.get pipeline handler-name)]
     (safe-execute
      (.channel pipeline)
      #(when-let [ctx (.context pipeline handler-name)]
         (.write ^ChannelOutboundHandler writer ctx msg p)))
     p)))

(defn invoke-write-all
  [^ChannelPipeline pipeline ^String handler-name frames]
  (when-let [writer (.get pipeline handler-name)]
    (let [flushed-p (DefaultChannelPromise. (.channel pipeline))
          written-p (DefaultChannelPromise. (.channel pipeline))
          complete (atom 0)]
      (safe-execute
       (.channel pipeline)
       #(when-let [ctx (.context pipeline handler-name)]
          (doseq [^Http2DataFrame frame frames]
            (let [p (DefaultChannelPromise. (.channel pipeline))]
              (.write ^ChannelOutboundHandler writer ctx frame p)
              (with-promise-listener p
                (fn []
                  (when (= (count frames)
                           (swap! complete inc))
                    (.setSuccess flushed-p))))))
          (.setSuccess written-p)))
      {:flushed flushed-p
       :written written-p})))

(defn invoke-channel-read
  [^ChannelPipeline pipeline ^String handler-name msg]
  (when-let [reader (.get pipeline handler-name)]
    (safe-execute
     (.channel pipeline)
     #(when-let [ctx (.context pipeline handler-name)]
        (.channelRead ^ChannelInboundHandler reader ctx msg)))))

(defn safe-remove-handler
  [^ChannelPipeline pipeline ^String handler-name]
  (safe-execute
   (.channel pipeline)
   #(when ((set (.names pipeline)) handler-name)
      (.remove pipeline handler-name))))

(defn safe-add-handler
  [^ChannelPipeline pipeline ^String handler-name ^ChannelHandler handler]
  (safe-execute
   (.channel pipeline)
   #(if (.get pipeline handler-name)
      (.replace pipeline handler-name handler-name handler)
      (.addLast pipeline handler-name handler))))

(definline release [x]
  `(io.netty.util.ReferenceCountUtil/release ~x))

(definline acquire [x]
  `(io.netty.util.ReferenceCountUtil/retain ~x))

(defn allocate [x]
  (if (instance? Channel x)
    (-> ^Channel x .alloc .ioBuffer)
    (-> ^ChannelHandlerContext x .alloc .ioBuffer)))

(defn append-to-buf! [^ByteBuf buf x]
  (cond
    (instance? array-class x)
    (.writeBytes buf ^bytes x)

    (instance? String x)
    (.writeBytes buf (.getBytes ^String x charset))

    (instance? ByteBuf x)
    (let [b (.writeBytes buf ^ByteBuf x)]
      (release x)
      b)

    :else
    (.writeBytes buf ^ByteBuf (to-byte-buf x))))

(defn- input-stream->byte-array
  [^InputStream is]
  (with-open [baos (java.io.ByteArrayOutputStream.)]
    (io/copy is baos)
    (.toByteArray baos)))

(defn to-byte-buf
  (^ByteBuf [x]
   (cond
     (nil? x)
     Unpooled/EMPTY_BUFFER

     (instance? array-class x)
     (Unpooled/copiedBuffer ^bytes x)

     (instance? String x)
     (-> ^String x (.getBytes charset)
         ByteBuffer/wrap
         Unpooled/wrappedBuffer)

     (instance? ByteBuffer x)
     (Unpooled/wrappedBuffer ^ByteBuffer x)

     (instance? ByteBuf x)
     x

     ;; TODO - this shouldn't read the entire file into memory
     (instance? File x)
     (recur
      (with-open [fis (io/input-stream x)]
        (input-stream->byte-array fis)))

     (instance? InputStream x)
     (recur (input-stream->byte-array x))

     :else (throw (ex-info "Unable to convert value to ByteBuf"
                           {:value x}))))
  (^ByteBuf [ch x]
   (if (nil? x)
     Unpooled/EMPTY_BUFFER
     (doto (allocate ch)
       (append-to-buf! x)))))

(defn leak-detector-level! [level]
  (ResourceLeakDetector/setLevel
   (case level
     :disabled ResourceLeakDetector$Level/DISABLED
     :simple ResourceLeakDetector$Level/SIMPLE
     :advanced ResourceLeakDetector$Level/ADVANCED
     :paranoid ResourceLeakDetector$Level/PARANOID)))

(defn slice-byte-buf
  [^ByteBuf byte-buf max-frame-size]
  (let [byte-count (.readableBytes byte-buf)
        slice-count (cond-> (Math/floor (/ byte-count max-frame-size))
                      (not (zero? (mod byte-count max-frame-size))) inc)]
    (mapv (fn [slice-index]
            (let [bytes-left (- byte-count (* slice-index max-frame-size))]
              (.slice byte-buf
                      (* slice-index max-frame-size)
                      (min max-frame-size
                           bytes-left))))
          (range slice-count))))

(defn byte-buf-to-http2-data-frames
  ([^Http2FrameStream stream ^ByteBuf byte-buf max-frame-size]
   (byte-buf-to-http2-data-frames stream byte-buf max-frame-size true))
  ([^Http2FrameStream stream ^ByteBuf byte-buf max-frame-size terminate?]
   (let [frames (slice-byte-buf byte-buf max-frame-size)]
     (map-indexed
      (fn [index ^ByteBuf frame]
        (doto (DefaultHttp2DataFrame.
               frame (boolean
                      (and terminate?
                           (= index (dec (count frames))))))
          (.stream stream)))
      frames))))

(defn byte-buf-to-bytes
  [^ByteBuf buf]
  (let [bytes (byte-array (.readableBytes buf))]
    (.readBytes buf bytes)
    bytes))
