;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.tcp
  (:require [fluxus.promise :as p]
            [fluxus.flow :as f]
            [vectio.netty :as n]
            [vectio.util.fides-tls :as fides])
  (:import [io.netty.bootstrap Bootstrap]
           [io.netty.buffer ByteBuf]
           [io.netty.channel
            ChannelHandler
            ChannelHandlerContext
            ChannelInitializer
            SimpleChannelInboundHandler
            ChannelPipeline]
           [io.netty.channel.nio NioEventLoopGroup]
           [io.netty.channel.socket SocketChannel]
           [io.netty.channel.socket.nio NioSocketChannel]
           [io.netty.handler.codec LengthFieldBasedFrameDecoder]
           [java.nio ByteOrder]
           [java.net InetSocketAddress]
           [io.netty.handler.ssl SslContextBuilder]
           [java.io InputStream]))

(declare channel-inbound-handler)

(def default-max-frame-length (* 16 1024)) ; 16K

(defn client
  [{:keys [host port tls]}]
  (let [group (NioEventLoopGroup.)
        [client internal] (f/entangled)
        client-flow (p/promise)
        client-pipeline (p/promise)]
    (f/on-close client (fn [_] (.sync (.shutdownGracefully group))))
    (try
      (let [client-bootstrap (Bootstrap.)
            ssl-context (when (not-empty tls)
                          (fides/validate tls :trust? true)
                          (let [{:keys [cert key trust]} (fides/->input-streams tls)]
                            (-> (SslContextBuilder/forClient)
                                (.keyManager ^InputStream cert ^InputStream key)
                                (.trustManager ^InputStream trust)
                                .build)))]
        (doto client-bootstrap
          (.group group)
          (.channel ^Class NioSocketChannel)
          (.remoteAddress (InetSocketAddress. ^String host ^long port))
          (.handler (proxy [ChannelInitializer] []
                      (initChannel [^SocketChannel socket-channel]
                        (let [pipeline (.pipeline socket-channel)]
                          (p/resolve! client-pipeline pipeline)
                          (when ssl-context
                            (.addLast pipeline "ssl-handler"
                                      (.newHandler ssl-context
                                                   (.alloc socket-channel)
                                                   host
                                                   (int port))))
                          (.addLast pipeline "inbound-handler"
                                    ^SimpleChannelInboundHandler (channel-inbound-handler internal)))))))
        (.sync (.connect client-bootstrap))
        (let [^ChannelPipeline pipeline @client-pipeline]
          (->> {:pipeline pipeline
                :add-first (fn [^String name ^ChannelHandler handler]
                             (if ssl-context
                               (.addAfter pipeline "ssl-handler" name handler)
                               (.addFirst pipeline ^String name handler)))
                :add-last (fn [^String name ^ChannelHandler handler]
                            (.addLast pipeline ^String name handler))}
               (with-meta client)
               (p/resolve! client-flow))))
      (catch Exception e
        (p/reject! client-flow e)))
    client-flow))

(defn length-based-decoder
  ^LengthFieldBasedFrameDecoder
  [{:keys [offset length max-frame-length]
    :or {max-frame-length default-max-frame-length}}]
  (LengthFieldBasedFrameDecoder. ByteOrder/LITTLE_ENDIAN max-frame-length offset length 0 0 true))

(defn add-handler
  [client name ^ChannelHandler handler
   & {:keys [position] :or {position :first}}]
  (let [^ChannelPipeline pipeline (-> client meta :pipeline)
        add-first (-> client meta :add-first)
        add-last (-> client meta :add-last)
        result (p/promise)]
    (n/safe-execute
     (.channel pipeline)
     (fn []
       (try
         (if (.get pipeline ^String name)
           (p/reject! result (Exception. "Handler already added"))
           (do
             (if (= :first position)
               (add-first ^String name handler)
               (add-last ^String name handler))
             (p/resolve! result true)))
         (catch Exception e
           (p/reject! result e)))))
    result))

(defn remove-handler
  [client handler-name]
  (let [^ChannelPipeline pipeline (-> client meta :pipeline)
        result (p/promise)]
    (n/safe-execute
     (.channel pipeline)
     (fn []
       (try
         (if ((set (.names pipeline)) handler-name)
           (do
             (.remove pipeline ^String handler-name)
             (p/resolve! result true))
           (p/resolve! result false))
         (catch Exception e
           (p/reject! result e)))))
    result))


;;; Private

(defn- channel-inbound-handler
  ^SimpleChannelInboundHandler [flow]
  (proxy [SimpleChannelInboundHandler] []
    (channelActive [^ChannelHandlerContext ctx]
      (f/consume
       #(let [byte-buf (n/to-byte-buf ctx %)]
          (n/safe-execute
           ctx (fn []
                 (.writeAndFlush ctx byte-buf))))
       flow))
    (channelRead0 [^ChannelHandlerContext ctx msg]
      (let [content ^ByteBuf msg
            bytes (byte-array (.readableBytes content))]
        (.readBytes content bytes)
        (f/put! flow bytes)))))
