;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.http
  (:require [fluxus.flow :as f]
            [fluxus.promise :as p]
            [utilis.js :as j]
            [utilis.string :refer [ensure-starts-with]]))

(set! *warn-on-infer* true)

(defn websocket-client
  [{:keys [socket-fn address on-close]}]
  (let [^js/WebSocket socket (if socket-fn
                               (socket-fn)
                               (js/WebSocket. address))
        [client-stream internal] (f/entangled)
        connection (p/promise)
        status (fn [e]
                 (let [reason (.-reason e)
                       reason-str (str reason)]
                   (cond-> {:reason reason
                            :code (.-code e)}
                     (boolean
                      (or (re-find #"(?i)java.security.cert.CertPathValidatorException" reason-str)
                          (re-find #"(?i)expected http 101 response" reason-str)
                          (re-find #"(?i)403" reason-str)))
                     (assoc :fatal? true))))]
    (set! (.-onopen socket)
          (fn [_]
            (p/resolve! connection client-stream)))
    (set! (.-onmessage socket)
          (fn [e]
            (f/put! internal (.-data e))))
    (set! (.-onclose socket)
          (fn [e]
            (f/close! internal)
            (when on-close
              (on-close (status e)))))
    (set! (.-onerror socket)
          (fn [e]
            (f/close! internal)
            (let [status (status e)]
              (when on-close
                (on-close status))
              (when-not (p/realized? connection)
                (p/reject! connection status)))))
    (f/on-close client-stream (fn [_] (.close socket)))
    (f/consume (fn [message] (.send socket ^js/String message)) internal)
    connection))

(defn websocket-origin
  ([] (websocket-origin nil))
  ([path]
   (when-let [location (j/get js/window :location)]
     (str (if (= "http:" (j/get location :protocol))
            "ws://"
            "wss://")
          (j/get location :host)
          (ensure-starts-with (str path) "/")))))
