;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.netty.h2.handlers.headers
  (:require [vectio.netty.h2.handlers.websocket :as ws]
            [vectio.netty :as n]
            [spectator.log :as log]
            [utilis.map :refer [map-keys assoc-if]]
            [utilis.types.number :refer [string->long]]
            [clojure.string :as st]
            [fluxus.promise :as p])
  (:import [java.util Map$Entry]
           [java.util.concurrent ExecutorService LinkedBlockingQueue]
           [java.io PipedOutputStream PipedInputStream File]
           [io.netty.buffer ByteBuf]
           [io.netty.handler.codec.http QueryStringDecoder]
           [io.netty.channel
            ChannelHandlerContext
            ChannelPipeline
            ChannelPromise]
           [io.netty.handler.stream ChunkedFile]
           [io.netty.handler.codec.http2
            Http2Error
            Http2HeadersFrame
            Http2Headers
            Http2FrameStream
            DefaultHttp2DataFrame
            DefaultHttp2Headers
            DefaultHttp2HeadersFrame
            DefaultHttp2ResetFrame
            Http2DataFrame]))

(declare headers-frame->request
         ensure-content-length
         setup-stream-window
         setup-upload-handler
         setup-ws-handler
         close-stream
         response->frames
         write-response-frames)

(defn handle?
  [msg]
  (instance? Http2HeadersFrame msg))

(defn handle
  [^ChannelHandlerContext ctx connection-state ^Http2HeadersFrame frame]
  (let [{:keys [handler exec-service peer-cert-chain]} @connection-state
        ^ExecutorService exec-service exec-service
        ^ChannelPipeline pipeline (.pipeline ctx)
        stream (.stream frame)
        stream-id (.id stream)
        request (-> frame
                    headers-frame->request
                    (assoc-if :peer-cert-chain peer-cert-chain))
        completed (p/promise)
        handle-error (fn [^Exception e]
                       (try
                         (log/error [::error e])
                         (write-response-frames
                          ctx connection-state stream-id
                          (response->frames
                           ctx
                           connection-state
                           stream
                           request
                           {:status 500
                            :completed completed}))
                         (.fireExceptionCaught ctx e)
                         (catch Exception e
                           (log/error [::error e]))))
        upload? (boolean (#{:put :post} (:request-method request)))
        ws? (ws/websocket-request? request)]
    (try
      (when (get-in @connection-state [:streams stream-id])
        ;; guard against stream reuse
        (close-stream ctx connection-state stream-id false))
      (setup-stream-window connection-state pipeline frame stream-id request)
      (let [upload-promise (when upload?
                             (setup-upload-handler
                              connection-state
                              stream-id
                              request))]
        (n/run exec-service
          (fn []
            (try
              (let [response (-> request
                                 handler
                                 ensure-content-length
                                 (assoc :completed completed))
                    response-frames (response->frames
                                     ctx
                                     connection-state
                                     stream
                                     request
                                     response)
                    write-response #(write-response-frames
                                     ctx
                                     connection-state
                                     stream-id
                                     response-frames)]
                (when ws?
                  (setup-ws-handler ctx connection-state frame stream-id response))
                (if upload?
                  (-> upload-promise
                      (p/then (fn [_] (write-response)))
                      (p/catch #(log/error [::handle :h2-upload-handler :error] %)))
                  (write-response)))
              (catch Exception e
                (handle-error e))))))
      (catch Exception e
        (handle-error e))))
  nil)


;;; Private

(defn- scheme
  [headers]
  (case (get headers ":scheme")
    "http" :http
    "https" :https
    "ws" :ws
    "wss" :wss))

(defn- parse-query-string
  [^String query-string]
  (->> (st/split query-string #"&")
       (remove empty?)
       (reduce (fn [query-params kv-string]
                 (let [[k v] (st/split kv-string #"\=")]
                   (assoc! query-params k v)))
               (transient {}))
       persistent!
       not-empty))

(defn- uri
  [headers]
  (-> headers
      (get ":path")
      (st/split #"\?")
      first))

(defn- request-method
  [headers]
  (case (get headers ":method")
    "GET" :get
    "HEAD" :head
    "POST" :post
    "PUT" :put
    "DELETE" :delete
    "CONNECT" :connect
    "OPTIONS" :options
    "TRACE" :trace
    "PATCH" :patch))

(defn- headers-frame->headers
  [^Http2HeadersFrame frame]
  (->> (.headers frame)
       .iterator
       iterator-seq
       (reduce (fn [m ^Map$Entry entry]
                 (assoc! m
                         (.toString (.getKey entry))
                         (.toString (.getValue entry))))
               (transient {}))
       persistent!))

(defn- create-body-input-stream
  [request]
  (let [content-length (get-in request [:headers "content-length"])
        content-length (when (string? content-length)
                         (string->long content-length))]
    (when (not content-length)
      (throw (ex-info "No content-length header provided in upload"
                      {:request request})))
    (if (pos? content-length)
      (let [in (PipedInputStream. (int content-length))]
        (assoc request
               :body in
               :body-output-stream (PipedOutputStream. in)))
      request)))

(defn- headers-frame->request
  [^Http2HeadersFrame frame]
  (let [headers (headers-frame->headers frame)
        _ (when (or (not (get headers ":path"))
                    (not (get headers ":scheme"))
                    (not (get headers ":authority")))
            (throw (ex-info "An http2 headers frame must include a :path, :scheme and :authority."
                            {:headers headers})))
        ^String path (get headers ":path")
        request-method (request-method headers)
        query-string (.rawQuery (QueryStringDecoder. path))]
    (cond-> {:headers headers
             :protocol-version "http/2"
             :uri (uri headers)
             :path path
             :scheme (scheme headers)
             :query-string query-string
             :request-method request-method
             :query-params (when query-string (parse-query-string query-string))}
      (#{:put :post} request-method) (create-body-input-stream))))

(defn- ensure-content-length
  [{:keys [body] :as response}]
  (let [response (or (not-empty response)
                     {:status 404})]
    (if-not (get-in response [:headers "Content-Length"])
      (cond-> response
        (or (bytes? body) (string? body))
        (assoc-in [:headers "Content-Length"] (count body))
        (and (instance? java.io.File body)
             (.exists ^java.io.File body))
        (assoc-in [:headers "Content-Length"] (.length ^java.io.File body)))
      response)))

(defn- close-stream
  ([^ChannelHandlerContext ctx connection-state stream-id]
   (close-stream ctx connection-state stream-id true))
  ([^ChannelHandlerContext ctx connection-state stream-id send-close?]
   (when-let [close (get-in @connection-state [:streams stream-id :close])]
     (n/safe-execute
      ctx (fn []
            (close :send-close send-close?))))))

(defn add-all-headers
  [^Http2Headers headers headers-map]
  (doseq [[^String k v] (map-keys #(st/lower-case
                                    (str (if (keyword? %)
                                           (name %)
                                           %)))
                                  headers-map)]
    (.add headers k (str v)))
  headers)

(defn response->headers
  [{:keys [status headers]}]
  (doto (DefaultHttp2Headers.)
    (.status (str status))
    (add-all-headers headers)))

(defn response->response-headers-frame
  ^Http2HeadersFrame [_connection-state ^Http2FrameStream stream request {:keys [body] :as response} response-data-frames]
  (-> (response->headers response)
      (DefaultHttp2HeadersFrame.
       (boolean
        (and (not (ws/websocket-request? request))
             (or (and (not (instance? LinkedBlockingQueue response-data-frames))
                      (zero? (count response-data-frames)))
                 (nil? body)
                 (and (instance? File body)
                      (zero? (.length ^File body)))))))
      (.stream stream)))

(defn response->response-data-frames
  [connection-state ^ChannelHandlerContext ctx ^Http2FrameStream stream _request {:keys [body]}]
  (when (not (nil? body))
    (let [max-frame-size (get-in @connection-state [:settings :settings-max-frame-size])]
      (if (instance? File body)
        (let [^File file body]
          (when (pos? (.length file))
            (let [queue (LinkedBlockingQueue. 100)
                  chunked-file (ChunkedFile. ^File file (long max-frame-size))
                  allocator (.alloc ctx)]
              (future
                (try (loop []
                       (let [chunk (.readChunk chunked-file allocator)
                             end? (boolean (.isEndOfInput chunked-file))
                             frame (-> chunk
                                       (DefaultHttp2DataFrame. end?)
                                       (.stream stream))]
                         (.put queue frame)
                         (if end?
                           (.close chunked-file)
                           (recur))))
                     (catch Exception e
                       (log/error [e]))))
              queue)))
        (let [^ByteBuf byte-buf (n/to-byte-buf ctx body)
              frames (->> max-frame-size
                          (n/byte-buf-to-http2-data-frames stream byte-buf)
                          (mapv n/acquire))]
          (n/release byte-buf)
          frames)))))

(defn response->frames
  [^ChannelHandlerContext ctx connection-state ^Http2FrameStream stream request response]
  (let [data-frames (response->response-data-frames
                     connection-state
                     ctx
                     stream
                     request
                     response)]
    {:data data-frames
     :headers (response->response-headers-frame
               connection-state
               stream
               request
               response
               data-frames)
     :response response
     :completed (:completed response)
     :byte-count (reduce (fn [c ^Http2DataFrame frame]
                           (+ c (.initialFlowControlledBytes frame)))
                         0
                         data-frames)}))

(defn- write-response-frames
  [^ChannelHandlerContext ctx connection-state stream-id response-frames]
  (let [^Http2HeadersFrame response-headers-frame (:headers response-frames)
        response-data-frames (:data response-frames)
        {:keys [write-frame]} @connection-state
        {:keys [closed?]} (get-in @connection-state [:streams stream-id])]
    (try
      (let [byte-flush-threshold 1E5
            byte-counter (volatile! 0)
            bytes-sent (volatile! 0)
            close-stream (fn [^ChannelPromise p]
                           (if p
                             (n/with-promise-listener p
                               (fn []
                                 (close-stream ctx connection-state stream-id true)))
                             (close-stream ctx connection-state stream-id false)))
            body? (not (.isEndStream response-headers-frame))
            blocking-safe-execute (fn [f]
                                    (let [ran (p/promise)
                                          result (atom nil)]
                                      (n/safe-execute
                                       ctx
                                       (fn []
                                         (try (reset! result (f))
                                              (p/resolve! ran true)
                                              (catch Exception e
                                                (p/reject! ran e)))))
                                      @ran
                                      @result))
            write-header (fn []
                           (blocking-safe-execute
                            (fn []
                              (let [^ChannelPromise p (write-frame response-headers-frame)]
                                (when (not body?)
                                  (close-stream p))))))
            write-body-chunk (fn [chunk final?]
                               (blocking-safe-execute
                                (fn []
                                  (let [result (atom nil)]
                                    (doseq [^Http2DataFrame frame chunk]
                                      (let [^ChannelPromise p (write-frame frame)
                                            frame-bytes (.initialFlowControlledBytes ^Http2DataFrame frame)]
                                        (vswap! byte-counter + frame-bytes)
                                        (vswap! bytes-sent + frame-bytes)
                                        (when (> @byte-counter byte-flush-threshold)
                                          (vreset! byte-counter 0)
                                          (.flush ctx)
                                          (reset! result p))
                                        (when (.isEndStream frame)
                                          (close-stream p))))
                                    (when final?
                                      (.flush ctx))
                                    @result))))]
        (write-header)
        (when body?
          (if (instance? LinkedBlockingQueue response-data-frames)
            (loop []
              (when-let [^Http2DataFrame frame (.take ^LinkedBlockingQueue response-data-frames)]
                (let [final? (.isEndStream frame)
                      ^ChannelPromise p (write-body-chunk [frame] final?)]
                  (when (and (not final?)(not @closed?))
                    (when p (.awaitUninterruptibly p 60000))
                    (recur)))))
            (let [chunks (partition-all 100 response-data-frames)]
              (doseq [[idx chunk] (map-indexed vector chunks)]
                (when (not @closed?)
                  (let [final? (= idx (dec (count chunks)))
                        ^ChannelPromise p (write-body-chunk chunk final?)]
                    (when p (.awaitUninterruptibly p 60000)))))))))
      (catch Exception e
        (log/error ["Exception occurred writing response" e]))
      (finally
        (p/resolve! (:completed response-frames) true)))))

(defn- setup-stream-window
  [connection-state ^ChannelPipeline pipeline ^Http2HeadersFrame msg stream-id request]
  (let [window-size (get-in @connection-state [:settings :settings-initial-window-size])
        {:keys [write-frame]} @connection-state
        backlog (volatile! [])
        closed? (atom false)
        send-close-frame (fn []
                           (try
                             (-> Http2Error/STREAM_CLOSED
                                 (DefaultHttp2ResetFrame.)
                                 (.stream (.stream msg))
                                 write-frame)
                             (catch Exception e
                               (log/error "Exception occurred in close-handler" e))))
        cleanup-handlers (atom [(fn []
                                  (doseq [[_ ^Http2DataFrame msg _] @backlog]
                                    (try (n/release msg)
                                         (catch Exception e
                                           (log/error "Exception releasing message from backlog" e))))
                                  (vreset! backlog nil))])
        handle-close (fn []
                       (when (not @closed?)
                         (reset! closed? true)
                         (doseq [handler @cleanup-handlers]
                           (try
                             (handler)
                             (catch Exception e
                               (log/error "Exception occurred in cleanup handler" e))))
                         (swap! connection-state update :streams dissoc stream-id)))]
    (n/with-promise-listener (.closeFuture (.channel pipeline))
      (fn []
        (handle-close)))
    (swap! connection-state update-in
           [:streams stream-id]
           (fn [stream]
             (assoc stream
                    :request request
                    :window (volatile! window-size)
                    :backlog backlog
                    :cleanup-handlers cleanup-handlers
                    :closed? closed?
                    :close (fn [& {:keys [send-close]
                                  :or {send-close true}}]
                             (when send-close
                               (send-close-frame))
                             (handle-close)))))))

(defn- setup-upload-handler
  [connection-state stream-id request]
  (let [{:keys [body-output-stream]} request]
    (when body-output-stream
      (let [^PipedOutputStream body-output-stream body-output-stream
            close-bos (fn []
                        (try (.close body-output-stream)
                             (catch Exception e
                               (log/error "Exception occurred closing body output stream" e))))
            done-p (p/promise)
            upload-total (atom 0)]
        (-> @connection-state
            (get-in [:streams stream-id :cleanup-handlers])
            (swap! (fn [handlers]
                     (conj (vec handlers)
                           (fn []
                             (close-bos)
                             (when-let [is (:body request)]
                               (try (.close ^java.io.InputStream is)
                                    (catch Exception _e)))
                             (try (when (not (p/realized? done-p))
                                    (p/reject! done-p :connection-closed))
                                  (catch Exception e
                                    (log/error "Exception occurred in cleanup handler" e))))))))
        (swap! connection-state assoc-in
               [:streams stream-id :data-frame-handler]
               (fn [^Http2DataFrame frame]
                 (let [^ByteBuf buf (.content frame)
                       bytes (byte-array (.readableBytes buf))]
                   (.readBytes buf bytes)
                   (swap! upload-total + (count bytes))
                   (.write body-output-stream bytes 0 (count bytes))
                   (.flush body-output-stream)
                   (when (.isEndStream frame)
                     (close-bos)
                     (p/resolve! done-p :done)))))
        done-p))))

(defn setup-ws-handler
  [^ChannelHandlerContext ctx connection-state ^Http2HeadersFrame msg stream-id response]
  (when-let [data-frame-handler (ws/init-handler ctx connection-state msg response)]
    (swap! connection-state assoc-in
           [:streams stream-id :data-frame-handler]
           data-frame-handler)))
