;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.udp
  (:require [fluxus.promise :as p]
            [fluxus.flow :as f]
            [vectio.netty :as n]
            [vectio.netty.server :as server])
  (:import [io.netty.bootstrap Bootstrap]
           [io.netty.buffer ByteBuf]
           [io.netty.channel
            ChannelHandler
            ChannelHandlerContext
            ChannelInitializer
            SimpleChannelInboundHandler
            ChannelPipeline]
           [io.netty.channel.nio NioEventLoopGroup]
           [io.netty.channel.socket DatagramChannel DatagramPacket]
           [io.netty.channel.socket.nio NioDatagramChannel]
           [java.util.concurrent ExecutorService]
           [java.net InetSocketAddress]))

(declare channel-inbound-handler)

(defn client
  [{:keys [host port]}]
  (let [group (NioEventLoopGroup.)
        [client internal] (f/entangled)
        client-flow (p/promise)
        client-pipeline (p/promise)]
    (f/on-close client (fn [_] (.shutdownGracefully group)))
    (try
      (let [client-bootstrap (Bootstrap.)]
        (doto client-bootstrap
          (.group group)
          (.channel ^Class NioDatagramChannel)
          (.remoteAddress (InetSocketAddress. ^String host ^long port))
          (.handler (proxy [ChannelInitializer] []
                      (initChannel [^DatagramChannel datagram-channel]
                        (let [pipeline (.pipeline datagram-channel)]
                          (p/resolve! client-pipeline pipeline)
                          (.addLast pipeline "inbound-handler"
                                    ^SimpleChannelInboundHandler (channel-inbound-handler internal)))))))
        (.sync (.connect client-bootstrap))
        (let [^ChannelPipeline pipeline @client-pipeline]
          (->> {:pipeline pipeline
                :add-first (fn [^String name ^ChannelHandler handler]
                             (.addFirst pipeline ^String name handler))
                :add-last (fn [^String name ^ChannelHandler handler]
                            (.addLast pipeline ^String name handler))}
               (with-meta client)
               (p/resolve! client-flow))))
      (catch Exception e
        (p/reject! client-flow e)))
    client-flow))

(defn add-handler
  [client name ^ChannelHandler handler
   & {:keys [position] :or {position :first}}]
  (let [{:keys [pipeline add-first add-last]} (meta client)
        ^ChannelPipeline pipeline pipeline
        result (p/promise)]
    (n/safe-execute
     (.channel pipeline)
     (fn []
       (try
         (if (.get pipeline ^String name)
           (p/reject! result (Exception. "Handler already added"))
           (do
             (if (= :first position)
               (add-first ^String name handler)
               (add-last ^String name handler))
             (p/resolve! result true)))
         (catch Exception e
           (p/reject! result e)))))
    result))

(defn remove-handler
  [client handler-name]
  (let [^ChannelPipeline pipeline (-> client
                                      meta
                                      :pipeline)
        result (p/promise)]
    (n/safe-execute
     (.channel pipeline)
     (fn []
       (try
         (if ((set (.names pipeline)) handler-name)
           (do
             (.remove pipeline ^String handler-name)
             (p/resolve! result true))
           (p/resolve! result false))
         (catch Exception e
           (p/reject! result e)))))
    result))

(defn server
  [{:keys [host port handler channel-handlers leak-detector-level]
    :or {leak-detector-level :disabled}}]
  (server/start-udp-server
   {:leak-detector-level leak-detector-level
    :init-channel (fn [^ExecutorService exec-service ^DatagramChannel ch]
                    (let [handle-message (fn [^bytes message]
                                           (handler message))]
                      (doseq [{:keys [name handler]} channel-handlers]
                        (.addFirst (.pipeline ch)
                                   ^String name
                                   ^ChannelHandler (handler)))
                      (doto (.pipeline ch)
                        (.addLast "message-handler"
                                  (proxy [SimpleChannelInboundHandler] []
                                    (channelRead0 [^ChannelHandlerContext ctx ^DatagramPacket msg]
                                      (n/run exec-service handle-message
                                        (n/byte-buf-to-bytes (.content msg))))
                                    (channelComplete [^ChannelHandlerContext ctx]
                                      (.flush ctx)))))))
    :socket-address (InetSocketAddress. ^String host ^int port)}))

;;; Private

(defn- channel-inbound-handler
  ^SimpleChannelInboundHandler [flow]
  (proxy [SimpleChannelInboundHandler] []
    (channelActive [^ChannelHandlerContext ctx]
      (f/consume
       #(let [byte-buf (n/to-byte-buf ctx %)]
          (n/safe-execute
           ctx (fn []
                 (.writeAndFlush ctx byte-buf))))
       flow))
    (channelRead0 [^ChannelHandlerContext ctx msg]
      (let [content ^ByteBuf msg
            bytes (byte-array (.readableBytes content))]
        (.readBytes content bytes)
        (f/put! flow bytes)))))
