;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns dev
  (:require [vectio.http :as http]
            [fides.tls :as tls]
            [reitit.ring :refer [ring-handler router]]
            [clojure.java.io :as io]

            [spectator.log :as log]
            [spectator.logger.console :as console]

            [integrant.core :as ig]
            [integrant.repl :refer [clear go halt init reset reset-all]]
            [integrant.repl.state :refer [system]]
            [fluxus.flow :as f]
            [fluxus.promise :as p]))

(log/set-level! :trace)
(log/set-logger! (console/logger {}))
(log/set-ns-level! "org.eclipse.jetty" :debug)

(f/set-default-on-error! (fn [& args] (log/error [:fluxus.flow/error args])))

(defn ws-handler
  [request]
  (try
    (when-let [{:keys [flow response]} (http/websocket-upgrade request)]
      (log/trace [:vectio.handlers/ws :response response :flow flow])
      (f/on-close flow
                  (fn [_]
                    (log/trace [:vectio.handlers/ws :closed flow])))
      (f/consume (fn [m]
                   (f/put! flow (str "echo: " m))
                   (log/trace [:vectio.handlers/ws :< m]))
                 flow)
      (f/put! flow "server-handshake")
      response)
    (catch Throwable e
      (log/error [:vectio.handler/ws] e)
      (throw e))))

(def config
  (let [cert-dir (str (System/getProperty "user.home") "/.7t/web/creds/")]
    {:structor.builder/watcher
     {:build-id :dev}
     :vectio.http/server
     {:host "localhost"
      :port 5443
      :tls {:cred (tls/pem-> {:cert (io/file (str cert-dir "server.crt"))
                              :key (io/file (str cert-dir "server.key"))})
            :trust (tls/pem-> {:cert (io/file (str cert-dir "ca.crt"))})}
      :handlers {:static
                 [["/static" (-> (io/file "dev") .toURI .toURL)]
                  ["/" (io/resource "public")]]
                 :ring
                 (ring-handler
                  (router
                   [["/ws" ws-handler]
                    ["/foo"
                     {:get {:handler
                            (fn [_req]
                              {:body "foo" :status 200})}}]]))}}}))

(ig/load-namespaces config)

(integrant.repl/set-prep! (constantly config))
