;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.http
  (:refer-clojure :exclude [get])
  (:require [vectio.jetty.handler :as handler]
            [vectio.jetty.websocket :as websocket]
            [fides.store :as keystore]
            [fluxus.flow :as f]
            [fluxus.promise :as p]
            [utilis.fs :as fs]
            [clojure.java.io :as io]
            [integrant.core :as ig]
            [spectator.log :as log])
  (:import  [org.eclipse.jetty.server
             Server ServerConnector
             ConnectionFactory
             HttpConfiguration HttpConnectionFactory
             SecureRequestCustomizer]
            [org.eclipse.jetty.http2.server
             HTTP2ServerConnectionFactory]
            [org.eclipse.jetty.http3.server
             HTTP3ServerConnectionFactory]
            [org.eclipse.jetty.alpn.server ALPNServerConnectionFactory]
            [org.eclipse.jetty.quic.server ServerQuicConfiguration QuicServerConnector]
            [org.eclipse.jetty.client HttpClient]
            [org.eclipse.jetty.http HttpMethod HttpFields$Mutable]
            [org.eclipse.jetty.util.ssl SslContextFactory$Server]
            [org.eclipse.jetty.util.thread QueuedThreadPool]
            [java.util.concurrent Executors]
            [java.nio.file Path]
            [java.io Closeable]))

;; TODO: Add client and websocket-client

(declare server)

(defmethod ig/init-key :vectio.http/server
  [_ opts]
  {:server (server opts)})

(defmethod ig/halt-key! :vectio.http/server
  [_ {:keys [server]}]
  (some-> ^Closeable server .close))

(defn server
  [{:keys [host port handlers tls authenticator cred-work-dir]
    :or {host "0.0.0.0"}
    :as server-opts}]
  (when (not port)
    (throw (ex-info ":vectio.http/server port is mandatory"
                    server-opts)))
  (when (not tls)
    (throw (ex-info ":vectio.http/server tls is mandatory"
                    server-opts)))
  (when-not (not-empty handlers)
    (throw (ex-info ":vectio.http/server handlers are mandatory"
                    server-opts)))
  (let [cred-work-dir (or cred-work-dir ".")
        creds-dir (let [creds-dir (fs/create-temp-dir cred-work-dir "creds-")]
                    (fs/mkdir creds-dir :recursive true)
                    (.deleteOnExit (io/file creds-dir))
                    creds-dir)
        thread-pool
        (doto (QueuedThreadPool.)
          (.setVirtualThreadsExecutor
           (Executors/newVirtualThreadPerTaskExecutor)))

        ^Server
        server (doto (Server. thread-pool)
                 (.setDefaultHandler (handler/default)))

        http-config
        (doto (HttpConfiguration.)
          (.addCustomizer (SecureRequestCustomizer.))
          (.setSendServerVersion false)
          (.setSendDateHeader false))

        http1 (HttpConnectionFactory. http-config)
        http2 (doto (HTTP2ServerConnectionFactory. http-config)
                (.setConnectProtocolEnabled true))

        ^ConnectionFactory alpn
        (doto (ALPNServerConnectionFactory. "h2,http/1.1")
          (.setDefaultProtocol "http/1.1"))

        ^SslContextFactory$Server
        ssl-context-factory
        (doto (SslContextFactory$Server.)
          (.setKeyStorePassword "secret")
          (.setKeyStore (keystore/create "secret" (:cred tls))))

        ssl-context-factory
        (if (:trust tls)
          (doto ssl-context-factory
            (.setNeedClientAuth true)
            (.setTrustStorePassword "secret")
            (.setTrustStore (keystore/create "secret" (:trust tls))))
          ssl-context-factory)

        tcp-connector
        (doto (ServerConnector.
               server
               ssl-context-factory
               ^"[Lorg.eclipse.jetty.server.ConnectionFactory;"
               (into-array ConnectionFactory [alpn http2 http1]))
          (.setHost host)
          (.setPort port)
          #_(.setIdleTimeout (* 3600 1000)))

        ^ServerQuicConfiguration
        quic-config
        (doto (ServerQuicConfiguration.
               ssl-context-factory
               (Path/of creds-dir (into-array String [])))
          (.setMaxBidirectionalRemoteStreams 1024))

        quic-connector
        (doto (QuicServerConnector.
               server
               quic-config
               ^"[Lorg.eclipse.jetty.server.ConnectionFactory;"
               (into-array HTTP3ServerConnectionFactory [(HTTP3ServerConnectionFactory. quic-config http-config)]))
          (.setHost host)
          (.setPort port)
          #_(.setIdleTimeout (* 3600 1000)))] ; 1 hour

    (doto server
      (.addConnector quic-connector)
      (.addConnector tcp-connector)
      (.setHandler (handler/->handler server handlers tls authenticator)))

    (.start server)
    (proxy [Closeable] []
      (close []
        (.stop server)
        (fs/rm creds-dir :recursive true)))))

(defn websocket-upgrade
  [request]
  (log/trace [:vectio.websocket/upgrade
              (assoc request
                     :reitit.core/match :log/removed
                     :reitit.core/router :log/removed)])
  (let [[message-flow internal] (f/entangled)
        response (websocket/upgrade
                  (:jetty request)
                  {:on-open
                   (fn [{:keys [send close]}]
                     (log/trace [:vectio.websocket/open])
                     (f/on-close internal (fn [_] (close)))
                     (f/consume send internal))
                   :on-close
                   (fn
                     ([]
                      (log/trace [:vectio.websocket/close])
                      (f/close! internal))
                     ([status reason]
                      (log/trace [:vectio.websocket/close status reason])
                      (f/close! internal)))
                   :on-error
                   (fn [^Throwable e]
                     (log/trace [:vectio.websocket/error] e))
                   :on-text-message
                   (fn [message]
                     @(f/put! internal message))
                   :on-binary-message
                   (fn [message]
                     @(f/put! internal message))
                   :on-ping-message
                   (fn [& _]
                     (log/trace [:vectio.websocket/ping])
                     @(f/put! internal :ping))
                   :on-pong-message
                   (fn [& _]
                     (log/trace [:vectio.websocket/pong])
                     @(f/put! internal :pong))}
                  {:version (get-in request [:headers :sec-websocket-version])
                   :extensions (get-in request [:headers :sec-websocket-extensions])})]
    (if response
      {:flow message-flow
       :response response}
      (do (f/close! internal) nil))))

(defn client
  (^HttpClient [] (client {}))
  (^HttpClient [{:keys [tls ssl-context timeout follow-redirects]
                 :or {follow-redirects false}}]
   (doto (proxy [HttpClient Closeable] []
           (close []
             (.stop ^HttpClient this)))
     (.setFollowRedirects follow-redirects)
     (.start))))

(declare send-request)

(defn get
  ([client url] (get client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :get))))

(defn post
  ([client url] (post client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :post))))

(defn put
  ([client url] (put client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :put))))

(defn patch
  ([client url] (patch client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :patch))))

(defn delete
  ([client url] (delete client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :delete))))

(defn head
  ([client url] (head client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :head))))

(defn options
  ([client url] (options client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :options))))


;;; Private

(defn- parse-response
  [response]
  #_{:url (str (.uri response))
     :status (.statusCode response)
     :body (.body response)
     :headers (->> (.map (.headers response))
                   (map (fn [[k v]]
                          (let [v (cond-> (if (> (count v) 1) v (first v))
                                    (= "content-length" k) string->long)]
                            [(keyword k) v])))
                   (into {}))})

(defn- send-request
  [^HttpClient client url request]
  (let [{:keys [request-method headers body-as body timeout]
         :or {body-as :input-stream}} request
        request (doto (.newRequest client url)
                  (.method (case request-method
                             :get HttpMethod/GET
                             :post HttpMethod/POST
                             :put HttpMethod/PUT
                             :patch HttpMethod/PATCH
                             :delete HttpMethod/DELETE
                             :head HttpMethod/HEAD
                             :options HttpMethod/OPTIONS))
                  (.headers
                   (fn [^HttpFields$Mutable request-headers]
                     (doseq [[key value] headers]
                       (.put request-headers key value)))))
        response (p/promise)]
    (p/resolve! response (.send request))
    response))
