;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.http
  (:refer-clojure :exclude [get])
  (:require
   [clojure.java.io :as io]
   [fides.store :as keystore]
   [fluxus.promise :as p]
   [integrant.core :as ig]
   [tempus.duration :as td]
   [utilis.fs :as fs]
   [vectio.jetty.handler :as handler])
  (:import
   [java.io Closeable]
   [java.nio.file Path]
   [java.util.concurrent Executors]
   [org.eclipse.jetty.alpn.server ALPNServerConnectionFactory]
   [org.eclipse.jetty.client HttpClient]
   [org.eclipse.jetty.http HttpFields$Mutable HttpMethod]
   [org.eclipse.jetty.http2.server
    HTTP2ServerConnectionFactory]
   [org.eclipse.jetty.http3.server
    HTTP3ServerConnectionFactory]
   [org.eclipse.jetty.quic.server QuicServerConnector ServerQuicConfiguration]
   [org.eclipse.jetty.server
    ConnectionFactory
    HttpConfiguration
    HttpConnectionFactory
    SecureRequestCustomizer
    Server
    ServerConnector]
   [org.eclipse.jetty.util.ssl SslContextFactory$Server]
   [org.eclipse.jetty.util.thread QueuedThreadPool]))

(declare server)

(defmethod ig/init-key :vectio.http/server
  [_ opts]
  {:server (server opts)})

(defmethod ig/halt-key! :vectio.http/server
  [_ {:keys [server]}]
  (some-> ^Closeable server .close))

(defn server
  [{:keys [host port handlers tls authenticator cred-work-dir idle-timeout]
    :or {host "0.0.0.0"
         idle-timeout (td/hours 1)}
    :as server-opts}]
  (when (not port)
    (throw (ex-info ":vectio.http/server port is mandatory"
                    server-opts)))
  (when (not tls)
    (throw (ex-info ":vectio.http/server tls is mandatory"
                    server-opts)))
  (when-not (not-empty handlers)
    (throw (ex-info ":vectio.http/server handlers are mandatory"
                    server-opts)))
  (let [cred-work-dir (or cred-work-dir ".")
        creds-dir (let [creds-dir (fs/create-temp-dir cred-work-dir "creds-")]
                    (fs/mkdir creds-dir :recursive true)
                    (.deleteOnExit (io/file creds-dir))
                    creds-dir)
        thread-pool
        (doto (QueuedThreadPool.)
          (.setVirtualThreadsExecutor
           (Executors/newVirtualThreadPerTaskExecutor)))

        ^Server
        server (Server. thread-pool)

        http-config
        (doto (HttpConfiguration.)
          (.addCustomizer (SecureRequestCustomizer.))
          (.setSendServerVersion false)
          (.setSendDateHeader false)
          (.setPersistentConnectionsEnabled true)
          (.setIdleTimeout (td/into :milliseconds idle-timeout)))

        http1 (HttpConnectionFactory. http-config)
        http2 (doto (HTTP2ServerConnectionFactory. http-config)
                (.setConnectProtocolEnabled true))

        ^ConnectionFactory alpn
        (doto (ALPNServerConnectionFactory. "h2,http/1.1")
          (.setDefaultProtocol "http/1.1"))

        ^SslContextFactory$Server
        ssl-context-factory
        (doto (SslContextFactory$Server.)
          (.setKeyStorePassword "secret")
          (.setKeyStore (keystore/create "secret" (:cred tls))))

        ssl-context-factory
        (if (:trust tls)
          (doto ssl-context-factory
            (.setNeedClientAuth true)
            (.setTrustStorePassword "secret")
            (.setTrustStore (keystore/create "secret" (:trust tls))))
          (doto ssl-context-factory
            (.setNeedClientAuth false)
            (.setWantClientAuth false)))

        tcp-connector
        (doto (ServerConnector.
               server
               ssl-context-factory
               ^"[Lorg.eclipse.jetty.server.ConnectionFactory;"
               (into-array ConnectionFactory [alpn http2 http1]))
          (.setHost host)
          (.setPort port)
          (.setIdleTimeout (td/into :milliseconds idle-timeout)))

        ^ServerQuicConfiguration
        quic-config
        (doto (ServerQuicConfiguration.
               ssl-context-factory
               (Path/of creds-dir (into-array String [])))
          (.setMaxBidirectionalRemoteStreams 1024))

        quic-connector
        (doto (QuicServerConnector.
               server
               quic-config
               ^"[Lorg.eclipse.jetty.server.ConnectionFactory;"
               (into-array HTTP3ServerConnectionFactory [(HTTP3ServerConnectionFactory. quic-config http-config)]))
          (.setHost host)
          (.setPort port)
          (.setIdleTimeout (td/into :milliseconds idle-timeout)))]

    (doto server
      (.addConnector quic-connector)
      (.addConnector tcp-connector)
      (.setDefaultHandler (handler/->default server handlers))
      (.setHandler (handler/->handler server handlers tls authenticator)))

    (.start server)
    (proxy [Closeable] []
      (close []
        (.stop server)
        (fs/rm creds-dir :recursive true)))))

(defn client
  (^HttpClient [] (client {}))
  (^HttpClient [{:keys [tls ssl-context timeout follow-redirects]
                 :or {follow-redirects false}}]
   (doto (proxy [HttpClient Closeable] []
           (close []
             (.stop ^HttpClient this)))
     (.setFollowRedirects follow-redirects)
     (.start))))

(declare send-request)

(defn get
  ([client url] (get client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :get))))

(defn post
  ([client url] (post client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :post))))

(defn put
  ([client url] (put client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :put))))

(defn patch
  ([client url] (patch client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :patch))))

(defn delete
  ([client url] (delete client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :delete))))

(defn head
  ([client url] (head client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :head))))

(defn options
  ([client url] (options client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :options))))


;;; Private

(defn- parse-response
  [response]
  #_{:url (str (.uri response))
     :status (.statusCode response)
     :body (.body response)
     :headers (->> (.map (.headers response))
                   (map (fn [[k v]]
                          (let [v (cond-> (if (> (count v) 1) v (first v))
                                    (= "content-length" k) string->long)]
                            [(keyword k) v])))
                   (into {}))})

(defn- send-request
  [^HttpClient client url request]
  (let [{:keys [request-method headers body-as body timeout]
         :or {body-as :input-stream}} request
        request (doto (.newRequest client url)
                  (.method (case request-method
                             :get HttpMethod/GET
                             :post HttpMethod/POST
                             :put HttpMethod/PUT
                             :patch HttpMethod/PATCH
                             :delete HttpMethod/DELETE
                             :head HttpMethod/HEAD
                             :options HttpMethod/OPTIONS))
                  (.headers
                   (fn [^HttpFields$Mutable request-headers]
                     (doseq [[key value] headers]
                       (.put request-headers key value)))))
        response (p/promise)]
    (p/resolve! response (.send request))
    response))
