;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.jetty.handler
  (:require
   [clojure.string :as st]
   [fides.tls :as tls]
   [spectator.log :as log]
   [tempus.core :as t]
   [tempus.duration :as td]
   [utilis.io :as io]
   [utilis.map :refer [compact]])
  (:import
   [java.net URL]
   [java.util Arrays List]
   [org.eclipse.jetty.http
    HttpField
    HttpFields$Mutable
    HttpHeader
    HttpURI
    ImmutableHttpFields]
   [org.eclipse.jetty.http HttpCookie HttpCookie$SameSite]
   [org.eclipse.jetty.io EndPoint$SslSessionData]
   [org.eclipse.jetty.server
    Handler
    Handler$Abstract
    Handler$Sequence
    Request
    Response
    Server]
   [org.eclipse.jetty.server.handler ContextHandler ResourceHandler]
   [org.eclipse.jetty.util Callback]
   [org.eclipse.jetty.util.resource ResourceFactory]
   [org.eclipse.jetty.websocket.server ServerWebSocketContainer]))

(declare mtls-interceptor resource-handlers ring-handler
         ->ring ring-> protocol method uri peer-cert-chain)

(defn ->default ^Handler
  [server {:keys [default] :as _handlers}]
  (if default
    (ring-handler server default)
    (proxy [Handler$Abstract] []
      (handle [^Request request ^Response response ^Callback callback]
        (let [ring-response {:status 404}]
          (log/trace [:vectio/default-handler :<> (uri request) ring-response])
          (ring-> {:protocol (protocol request)} ring-response request response callback)
          (.succeeded callback)
          true)))))

(defn ->handler ^Handler
  [server {:keys [ring static] :as _handlers} tls authenticator]
  (let [handlers (-> [(mtls-interceptor server tls authenticator)]
                     (concat (resource-handlers server static))
                     (concat [(when ring
                                (let [handler (ContextHandler.
                                               (ring-handler server ring)
                                               "/")]
                                  (ServerWebSocketContainer/ensure server handler)
                                  handler))]))]
    (Handler$Sequence.
     ^List (->> handlers
                (remove nil?)
                (into-array Handler)
                (Arrays/asList)))))

(defn mtls-interceptor ^Handler
  [_server tls authenticator]
  (proxy [Handler$Abstract] []
    (handle [^Request request ^Response response ^Callback callback]
      (try
        (let [unauthorized (fn []
                             (.setStatus response 407)
                             (.succeeded callback)
                             true)]
          (if (:trust tls)
            (if-let [peer-cert-chain (peer-cert-chain request)]
              (do
                (.setAttribute
                 request
                 "vectio.mtls/peer-cert-chain" peer-cert-chain)
                (if authenticator
                  (if-let [auth (authenticator peer-cert-chain (uri request))]
                    (do
                      (.setAttribute request "vectio.mtls/peer-auth" auth)
                      (log/trace [:vectio/mtls peer-cert-chain auth])
                      false)
                    (unauthorized))
                  false)
                false)
              (unauthorized))
            false))
        (catch Throwable e
          (log/error [:vectio.mtls/interceptor :error] e))))))

(defn resource-handlers ^Handler
  [^Server server resources]
  (map (fn [[^String uri ^URL path]]
         (ContextHandler.
          (doto (ResourceHandler.)
            (.setBaseResource (-> (ResourceFactory/of server)
                                  (.newResource path)))
            (.setUseFileMapping true)
            (.setDirAllowed false)
            (.setAcceptRanges true)
            (.setEtags true))
          uri))
       resources))

(defn ring-handler ^ContextHandler
  [_server ring-handler]
  (proxy [Handler$Abstract] []
    (handle [^Request request ^Response response ^Callback callback]
      (try
        (let [method (method request)
              uri (uri request)
              ring-request (->ring request response callback)
              respond (fn [ring-response]

                        (log/trace [:vectio/ring-handler :> method uri ring-response])
                        (when-not (.isCommitted response)
                          (ring-> ring-request ring-response request response callback)
                          (try
                            (.succeeded callback)
                            (catch Throwable e
                              (log/trace [:vectio.ring-handler/callback :error e]))))
                        true)]
          (try
            (log/trace [:vectio/ring-handler :< method uri ring-request])
            (if-let [ring-response (not-empty (ring-handler ring-request))]
              (respond (if (string? ring-response)
                         {:body ring-response}
                         ring-response))
              (do
                (log/trace [:vectio/ring-handler :< method uri :unmatched])
                false))
            (catch Throwable e
              (log/error [:vectio/ring-handler method uri :error] e)
              (respond {:status 500}))))
        (catch Throwable e
          (log/error [:vectio.ring/handler :error] e))))))


;;; Private

(defn- protocol
  [^Request request]
  (.getProtocol (.getConnectionMetaData request)))

(defn- method
  [^Request request]
  (keyword (st/lower-case (.getMethod request))))

(defn- uri
  [^Request request]
  (let [^HttpURI uri (.getHttpURI request)]
    (when uri (.getPath uri))))

(defn- peer-cert-chain
  [^Request request]
  (when-let [session-data (.getAttribute request EndPoint$SslSessionData/ATTRIBUTE)]
    (mapv tls/x509-> (.peerCertificates ^EndPoint$SslSessionData session-data))))

(defn- ->ring
  [^Request request ^Response response ^Callback callback]
  (let [^HttpURI uri (.getHttpURI request)
        ^ImmutableHttpFields headers (.getHeaders request)
        content-type (.. headers (get HttpHeader/CONTENT_TYPE))
        content-length (when-let [l (.. headers (get HttpHeader/CONTENT_LENGTH))]
                         (Integer/valueOf l))
        headers (->> (.getHeaders request)
                     (map (fn [^HttpField header]
                            [(keyword (.getLowerCaseName header))
                             (.getValue header)]))
                     (into {}))]
    {:jetty {:request request
             :response response
             :callback callback}
     :server-port (Request/getServerPort request)
     :server-name (Request/getServerName request)
     :remote-addr (Request/getRemoteAddr request)
     :uri (when uri (.getPath uri))
     :query-string (when uri (.getQuery uri))
     :scheme (when uri (keyword (.getScheme uri)))
     :request-method (method request)
     :protocol (protocol request)
     :headers headers
     :content-type content-type
     :content-length content-length
     :peer {:cert-chain (.getAttribute request "vectio.mtls/peer-cert-chain")
            :auth (.getAttribute request "vectio.mtls/peer-auth")}
     :cookies (->> (Request/getCookies request)
                   (map (fn [^HttpCookie c]
                          [(.getName c)
                           (compact
                            {:value (.getValue c)
                             :domain (.getDomain c)
                             :path (.getPath c)
                             :max-age (.getMaxAge c)
                             :expires (when-let [expires (.getExpires c)]
                                        (t/from :long (.toEpochMilli expires)))
                             :secure (.isSecure c)
                             :http-only (.isHttpOnly c)
                             :same-site (when-let [same-site (.getSameSite c)]
                                          ({HttpCookie$SameSite/STRICT :strict
                                            HttpCookie$SameSite/LAX :lax
                                            HttpCookie$SameSite/NONE :none}
                                           same-site))})]))
                   (into {}))
     :body (Request/asInputStream request)}))

(defn- ensure-content-length
  [{:keys [body] :as response}]
  (let [response (or (not-empty response)
                     {:status 404})]
    (if-not (get-in response [:headers "content-length"])
      (cond-> response
        (bytes? body)
        (assoc-in [:headers "content-length"] (count body))

        (string? body)
        (assoc-in [:headers "content-length"] (count (.getBytes ^String body)))

        (and (instance? java.io.File body)
             (.exists ^java.io.File body))
        (assoc-in [:headers "content-length"] (.length ^java.io.File body)))
      response)))

(defn- ring->
  [_ring-request ring-response
   ^Request request ^Response response  ^Callback _callback]
  (let [{:keys [status headers cookies body]} (-> ring-response
                                                  ensure-content-length)]
    (some->> status (.setStatus response))
    (let [^HttpFields$Mutable jetty-headers (.getHeaders response)]
      (doseq [[k v] headers]
        (let [k (cond-> k (keyword? k) clojure.core/name)]
          (cond
            (string? v)
            (.put jetty-headers ^String k ^String v)
            (integer? v)
            (.put jetty-headers ^String k ^long v)
            :else
            (.put jetty-headers
                  ^String k
                  ^List (->> v (into-array String) (Arrays/asList)))))))
    (doseq [[key {:keys [value max-age path domain secure same-site http-only]
                  :or {secure true
                       same-site :strict
                       http-only true}}] cookies]
      (Response/addCookie
       response
       (cond-> (HttpCookie/build ^String key ^String value)
         domain (.domain domain)
         path (.path path)
         max-age (.maxAge (td/into :seconds max-age))
         secure (.secure secure)
         http-only (.httpOnly http-only)
         same-site (.sameSite ({:strict HttpCookie$SameSite/STRICT
                                :lax HttpCookie$SameSite/LAX
                                :none HttpCookie$SameSite/NONE}
                               same-site))
         true .build)))
    (when body
      (io/copy (io/input-stream body)
               (Response/asBufferedOutputStream request response)))))
