;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.websocket
  (:require
   [vectio.jetty.websocket :as websocket]
   [fluxus.flow :as f]
   [spectator.log :as log])
  (:import
   [java.io Closeable]
   [java.net URI]
   [org.eclipse.jetty.client HttpClient]
   [org.eclipse.jetty.websocket.client WebSocketClient]
   [vectio.jetty.websocket Listener]))

(defn upgrade
  [request]
  (log/trace [:vectio.websocket/upgrade
              (assoc request
                     :reitit.core/match :log/removed
                     :reitit.core/router :log/removed)])
  (let [[message-flow internal] (f/entangled)
        response (websocket/upgrade
                  (:jetty request)
                  {:on-open
                   (fn [{:keys [send close]}]
                     (log/trace [:vectio.websocket/open])
                     (f/on-close internal (fn [_] (close)))
                     (f/consume send internal))
                   :on-close
                   (fn
                     ([]
                      (log/trace [:vectio.websocket/close])
                      (f/close! internal))
                     ([status reason]
                      (log/trace [:vectio.websocket/close status reason])
                      (f/close! internal)))
                   :on-error
                   (fn [^Throwable e]
                     (log/trace [:vectio.websocket/error] e))
                   :on-text-message
                   (fn [message]
                     (log/trace [:vectio.websocket/on-text-message message])
                     @(f/put! internal message))
                   :on-binary-message
                   (fn [message]
                     @(f/put! internal message))}
                  {:version (get-in request [:headers :sec-websocket-version])
                   :extensions (get-in request [:headers :sec-websocket-extensions])})]
    (log/trace [:vectio.websocket/upgrade :response response])
    (if response
      {:flow message-flow
       :response response}
      (do (f/close! internal) nil))))

(defn client ^WebSocketClient
  []
  (let [^HttpClient http-client (HttpClient.)]
    (doto (proxy [WebSocketClient Closeable] [http-client]
            (close []
              (.stop ^WebSocketClient this)))
      (.start))))

(defn connect
  [client uri]
  (let [uri (URI/create uri)
        [message-flow internal] (f/entangled)
        handlers {:on-open
                  (fn [{:keys [send close]}]
                    (log/trace [:vectio.websocket.client/open])
                    (f/on-close internal (fn [_] (close)))
                    (f/consume send internal))
                  :on-close
                  (fn
                    ([]
                     (log/trace [:vectio.websocket.client/close])
                     (f/close! internal))
                    ([status reason]
                     (log/trace [:vectio.websocket.client/close status reason])
                     (f/close! internal)))
                  :on-error
                  (fn [^Throwable e]
                    (log/trace [:vectio.websocket.client/error] e))
                  :on-text-message
                  (fn [message]
                    (log/trace [:vectio.websocket.client/on-text-message message])
                    @(f/put! internal message))
                  :on-binary-message
                  (fn [message]
                    @(f/put! internal message))}
        listener (Listener. handlers (atom -1) (atom nil))
        session (.connect client listener uri)]
    (if (.get session)
      {:flow message-flow}
      (do (f/close! internal) nil))))
