;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.http
  (:refer-clojure :exclude [get])
  (:require
   [clojure.string :as st]
   [fluxus.promise :as p]
   [utilis.js :as j]
   [utilis.map :refer [map-keys]]
   [utilis.string :refer [ensure-starts-with]]))

(defn origin
  ([] (origin nil))
  ([path]
   (when-let [location (j/get js/window :location)]
     (let [href (j/get location (if path :origin :href))
           prefix-path? (not (st/ends-with? href "/"))]
       (str href (cond-> path prefix-path? (ensure-starts-with "/")))))))

(declare send-request)

(defn send
  [request]
  (send-request request))

(defn get
  ([uri] (get uri nil))
  ([uri request]
   (send (assoc request :uri uri :request-method :get))))

(defn post
  ([uri] (post uri nil))
  ([uri request]
   (send (assoc request :uri uri :request-method :post))))

(defn put
  ([uri] (put uri nil))
  ([uri request]
   (send (assoc request :uri uri :request-method :put))))

(defn patch
  ([uri] (patch uri nil))
  ([uri request]
   (send (assoc request :uri uri :request-method :patch))))

(defn delete
  ([uri] (delete uri nil))
  ([uri request]
   (send (assoc request :uri uri :request-method :delete))))

(defn head
  ([uri] (head uri nil))
  ([uri request]
   (send (assoc request :uri uri :request-method :head))))

(defn options
  ([uri] (options uri nil))
  ([uri request]
   (send (assoc request :uri uri :request-method :options))))


;; Private

(defn- parse-response
  [{:keys [codec] :as _request} fetch-response]
  (let [response (p/promise)]
    (-> (j/call fetch-response (if (= :json codec) :json :text))
        (j/call
         :then
         (fn [body-text]
           (p/resolve!
            response
            {:uri (j/get fetch-response :url)
             :status (j/get fetch-response :status)
             :headers (-> (j/get fetch-response :headers)
                          (js->clj :keywordize-keys true))
             :body (cond-> body-text (= :json codec) (js->clj :keywordize-keys true))}))))
    response))

(defn- send-request
  [request]
  (let [{:keys [uri request-method headers body codec]} request
        method ({:get "GET"
                 :post "POST"
                 :put "PUT"
                 :patch "PATCH"
                 :delete "DELETE"
                 :head "HEAD"
                 :options "OPTIONS"} request-method)
        response-p (p/promise)]
    (-> (js/fetch uri (cond-> {:method method}
                        headers (assoc :headers (map-keys clojure.core/name headers))
                        body (assoc :body body)
                        true clj->js))
        (j/call
         :then
         (fn [fetch-response]
           (-> (parse-response request fetch-response)
               (p/then (partial p/resolve! response-p)))) )
        (j/call
         :catch
         (fn [fetch-response]
           (-> (parse-response request fetch-response)
               (p/then (partial p/reject! response-p))))))
    response-p))
