;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.http
  (:refer-clojure :exclude [get send])
  (:require
   [fides.store :as keystore]
   [fluxus.promise :as p]
   [integrant.core :as ig]
   [spectator.log :as log]
   [tempus.duration :as td]
   [utilis.fs :as fs]
   [utilis.io :as io]
   [utilis.types.number :refer [string->long]]
   [vectio.jetty.handler :as handler])
  (:import
   [java.io Closeable]
   [java.nio.file Path]
   [java.util.concurrent Executors]
   [org.eclipse.jetty.alpn.server ALPNServerConnectionFactory]
   [org.eclipse.jetty.client
    ContentResponse
    HttpClient
    InputStreamRequestContent
    Response]
   [org.eclipse.jetty.http
    HttpCookie
    HttpField
    HttpFields$Mutable
    HttpMethod]
   [org.eclipse.jetty.http2.server
    HTTP2ServerConnectionFactory]
   [org.eclipse.jetty.http3.server
    HTTP3ServerConnectionFactory]
   [org.eclipse.jetty.quic.server QuicServerConnector ServerQuicConfiguration]
   [org.eclipse.jetty.server
    ConnectionFactory
    HttpConfiguration
    HttpConnectionFactory
    SecureRequestCustomizer
    Server
    ServerConnector]
   [org.eclipse.jetty.util.ssl SslContextFactory$Server]
   [org.eclipse.jetty.util.thread QueuedThreadPool]))

(declare server)

(defmethod ig/init-key :vectio.http/server
  [_ opts]
  {:server (server opts)})

(defmethod ig/halt-key! :vectio.http/server
  [_ {:keys [server]}]
  (some-> ^Closeable server .close))

(defn server
  [{:keys [host port handlers tls authenticator cred-work-dir idle-timeout]
    :or {host "0.0.0.0"
         idle-timeout (td/hours 1)}
    :as server-opts}]
  (when (not port)
    (throw (ex-info ":vectio.http/server port is mandatory"
                    server-opts)))
  (when (not tls)
    (throw (ex-info ":vectio.http/server tls is mandatory"
                    server-opts)))
  (when-not (not-empty handlers)
    (throw (ex-info ":vectio.http/server handlers are mandatory"
                    server-opts)))
  (let [cred-work-dir (or cred-work-dir ".")
        creds-dir (let [creds-dir (fs/create-temp-dir cred-work-dir "creds-")]
                    (fs/mkdir creds-dir :recursive true)
                    (.addShutdownHook
                     (Runtime/getRuntime)
                     (Thread. ^Runnable #(try
                                           (fs/rm creds-dir :recursive true)
                                           (catch java.lang.Throwable e
                                             (log/error [:vectio.h3/creds :delete-error creds-dir] e)))))
                    creds-dir)
        thread-pool
        (doto (QueuedThreadPool.)
          (.setVirtualThreadsExecutor
           (Executors/newVirtualThreadPerTaskExecutor)))

        ^Server
        server (Server. thread-pool)

        http-config
        (doto (HttpConfiguration.)
          (.addCustomizer (SecureRequestCustomizer.))
          (.setSendServerVersion false)
          (.setSendDateHeader false)
          (.setPersistentConnectionsEnabled true)
          (.setIdleTimeout (td/into :milliseconds idle-timeout)))

        http1 (HttpConnectionFactory. http-config)
        http2 (doto (HTTP2ServerConnectionFactory. http-config)
                (.setConnectProtocolEnabled true))

        ^ConnectionFactory alpn
        (doto (ALPNServerConnectionFactory. "h2,http/1.1")
          (.setDefaultProtocol "http/1.1"))

        ^SslContextFactory$Server
        ssl-context-factory
        (doto (SslContextFactory$Server.)
          (.setKeyStorePassword "secret")
          (.setKeyStore (keystore/create "secret" (:cred tls))))

        ssl-context-factory
        (if (:trust tls)
          (doto ssl-context-factory
            (.setNeedClientAuth true)
            (.setTrustStorePassword "secret")
            (.setTrustStore (keystore/create "secret" (:trust tls))))
          (doto ssl-context-factory
            (.setNeedClientAuth false)
            (.setWantClientAuth false)))

        tcp-connector
        (doto (ServerConnector.
               server
               ssl-context-factory
               ^"[Lorg.eclipse.jetty.server.ConnectionFactory;"
               (into-array ConnectionFactory [alpn http2 http1]))
          (.setHost host)
          (.setPort port)
          (.setIdleTimeout (td/into :milliseconds idle-timeout)))

        ^ServerQuicConfiguration
        quic-config
        (doto (ServerQuicConfiguration.
               ssl-context-factory
               (Path/of creds-dir (into-array String [])))
          (.setMaxBidirectionalRemoteStreams 1024))

        quic-connector
        (doto (QuicServerConnector.
               server
               quic-config
               ^"[Lorg.eclipse.jetty.server.ConnectionFactory;"
               (into-array HTTP3ServerConnectionFactory [(HTTP3ServerConnectionFactory. quic-config http-config)]))
          (.setHost host)
          (.setPort port)
          (.setIdleTimeout (td/into :milliseconds idle-timeout)))]

    (doto server
      (.addConnector quic-connector)
      (.addConnector tcp-connector)
      (.setDefaultHandler (handler/->default server handlers))
      (.setHandler (handler/->handler server handlers tls authenticator)))

    (.start server)
    (proxy [Closeable] []
      (close []
        (.stop server)
        (fs/rm creds-dir :recursive true)))))

(defn client
  (^HttpClient [] (client {}))
  (^HttpClient [{:keys [tls ssl-context timeout follow-redirects]
                 :or {follow-redirects false}}]
   (doto (proxy [HttpClient Closeable] []
           (close []
             (.stop ^HttpClient this)))
     (.setFollowRedirects follow-redirects)
     (.start))))

(defn dispose
  [^HttpClient client]
  (.stop client))

(declare send-request)

(defn send
  [client request]
  (send-request client request))

(defn get
  ([client uri] (get client uri nil))
  ([client uri request]
   (send client (assoc request :uri uri :request-method :get))))

(defn post
  ([client uri] (post client uri nil))
  ([client uri request]
   (send client (assoc request :uri uri :request-method :post))))

(defn put
  ([client uri] (put client uri nil))
  ([client uri request]
   (send client (assoc request :uri uri :request-method :put))))

(defn patch
  ([client uri] (patch client uri nil))
  ([client uri request]
   (send client (assoc request :uri uri :request-method :patch))))

(defn delete
  ([client uri] (delete client uri nil))
  ([client uri request]
   (send client (assoc request :uri uri :request-method :delete))))

(defn head
  ([client uri] (head client uri nil))
  ([client uri request]
   (send client (assoc request :uri uri :request-method :head))))

(defn options
  ([client uri] (options client uri nil))
  ([client uri request]
   (send client (assoc request :uri uri :request-method :options))))


;;; Private

(defn- parse-response
  [uri ^Response response]
  (let [content (.getContent ^ContentResponse response)]
    (cond-> {:uri uri
             :protocol (.asString (.getVersion response))
             :status (.getStatus response)
             :headers (->> (.getHeaders response)
                           (map (fn [^HttpField header]
                                  (let [k (keyword (.getLowerCaseName header))]
                                    [k (cond-> (.getValue header)
                                         (= :content-length k) string->long)])))
                           (into {}))}
      (pos? (count content))
      (assoc :body (io/input-stream content)))))

(defn- send-request
  [^HttpClient client ring-request]
  (let [{:keys [uri request-method headers cookies body]} ring-request
        request (doto (.newRequest client ^String uri)
                  (.method (case request-method
                             :get HttpMethod/GET
                             :post HttpMethod/POST
                             :put HttpMethod/PUT
                             :patch HttpMethod/PATCH
                             :delete HttpMethod/DELETE
                             :head HttpMethod/HEAD
                             :options HttpMethod/OPTIONS))
                  (.headers
                   (fn [^HttpFields$Mutable request-headers]
                     (doseq [[key value] headers]
                       (.put request-headers ^String (clojure.core/name key) ^String (str value))))))
        response (p/promise)]
    (when cookies
      (doseq [[key {:keys [value]}] cookies]
        (.cookie request (HttpCookie/from ^String (clojure.core/name key) ^String (str value)))))
    (when body
      (.body request (InputStreamRequestContent. (io/input-stream body))))
    (p/resolve! response (parse-response uri (.send request)))
    response))
