(ns via-schema.core
  (:require [via-schema.syntax.ghostwheel :as gw]
            [malli.core :as m]
            [malli.error :as me]
            [malli.transform :as mt]
            [clojure.walk :refer [postwalk]]))

;;; Declarations

(declare tx-ret-schema remap-underscores coerce-body generate)

(defonce humanize-errors (atom true))

;;; API

(defn set-humanize-errors
  [humanize?]
  (reset! humanize-errors humanize?))

(defmacro >fn
  "Args are the same as >defn from aave (https://github.com/teknql/aave), but
  without a function name (as nothing will be defined at the namespace level)."
  {:arglists '([doc-string? attr-map? [params*] [schemas*]? body])}
  [& args]
  (let [cfg (-> (symbol (str "validate-" (gensym)))
                (cons args)
                (gw/parse))
        ret? (boolean (:ret-schema cfg))
        {:keys [ret-schema param-schema]
         :as cfg} (cond-> cfg
                    ret? (update :ret-schema remap-underscores)
                    (:param-schema cfg) (update :param-schema remap-underscores))
        cfg (cond-> cfg
              ret? (-> (assoc :orig-ret-schema ret-schema)
                       (update :ret-schema tx-ret-schema)))]
    `(generate ~cfg)))

;;; Implementation

(def underscore (symbol "_"))
(def underscore? (partial = underscore))

(defn- remap-underscores
  [x]
  (postwalk #(if (underscore? %) 'any? %) x))

(defn tx-ret-schema
  [schema]
  [:multi {::outstrument true
           :dispatch '(fn [result]
                        (if (and (map? result)
                                 (contains? result :via/reply))
                          (if (= 200 (-> result :via/reply :status))
                            :via/reply-ok
                            :via/reply-error)
                          :via/sub))}
   [:via/reply-ok [:map [:via/reply
                         [:map
                          [:status :int]
                          [:body {:optional true} schema]]]]]
   [:via/reply-error 'any?]
   [:via/sub schema]])

(def transformer
  (mt/transformer
   (mt/strip-extra-keys-transformer)
   (mt/default-value-transformer)
   (mt/json-transformer)))

(defmacro generate
  [{:keys [param-schema ret-schema name params+body]}]
  `(let [f# ~(concat (list 'fn name) params+body)
         continue-on-error# (fn [f#]
                              (fn [arg#]
                                (try (f# arg#)
                                     (catch Exception e#
                                       arg#))))
         [validate-params#
          explain-params#
          coerce-params#] (if ~param-schema
                            [(m/validator ~param-schema)
                             (m/explainer ~param-schema)
                             (->> transformer
                                  (m/decoder ~param-schema)
                                  (continue-on-error#))]
                            [(constantly true)
                             (constantly true)
                             identity])
         [validate-result#
          explain-result#
          coerce-body#] (if ~ret-schema
                          [(m/validator ~ret-schema)
                           (m/explainer ~ret-schema)
                           (->> transformer
                                (m/decoder ~ret-schema)
                                (continue-on-error#))]
                          [(constantly true)
                           (constantly true)
                           identity])
         wrap-errors# (fn [args# errors# schema#]
                        (let [event?# (boolean (map? (first args#)))]
                          (->> (when @humanize-errors
                                 {:schema schema#})
                               (merge {:query-v (if event?#
                                                  (vec (second args#))
                                                  (vec args#))
                                       :errors (cond-> errors#
                                                 @humanize-errors me/humanize)})
                               (clojure.walk/postwalk
                                (fn [x#]
                                  (cond
                                    (= any? x#) (symbol "_")
                                    (fn? x#) (str x#)
                                    :else x#))))))]
     (fn [& args#]
       (let [coerced-args# (->> args# (into []) coerce-params#)]
         (if (validate-params# coerced-args#)
           (let [result# (apply f# coerced-args#)]
             (if (validate-result# result#)
               (if (and (and (map? result#)
                             (contains? result# :via/reply))
                        (= 200 (:status (:via/reply result#))))
                 (update-in result# [:via/reply :body] coerce-body#)
                 (coerce-body# result#))
               (throw (->> ~ret-schema
                           (wrap-errors# coerced-args# (explain-result# result#))
                           (ex-info "Outstrement Error")))))
           (throw (->> ~param-schema
                       (wrap-errors# coerced-args# (explain-params# coerced-args#))
                       (ex-info "Instrument Error"))))))))
