;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.service
  (:refer-clojure :exclude [proxy])
  (:require
   #?(:clj [cognitect.transit :as transit-clj]
      :cljs [cognitect.transit :as transit-cljs])
   #?(:clj [clojure.pprint :refer [simple-dispatch]])
   [clojure.set :refer [map-invert rename-keys]]
   [fluxus.flow :as f]
   [fluxus.lock :refer [with-lock]]
   [fluxus.promise :as p]
   [fluxus.scheduler :as fs]
   [integrant.core :as ig]
   [signum.events :as se]
   [signum.fx :as sfx]
   [signum.subs :as ss]
   [spectator.log :as log]
   [tempus.core :as t]
   [tempus.duration :as td]
   [tempus.series :as ts]
   [tempus.transit :as tt]
   [utilis.jar :as jar]
   [utilis.string :as ust]
   [utilis.timer :as timer]
   [utilis.uuid :as uuid]
   [via.proxy :refer [#?(:cljs Proxy) proxy] :as vp]
   [via.util.ns :as ns])
  #?(:clj (:import
           [clojure.lang IMeta IObj]
           [java.io ByteArrayInputStream ByteArrayOutputStream]
           [java.util.concurrent.locks ReentrantLock]
           [via.proxy Proxy])))

#?(:cljs (set! *warn-on-infer* true))

(declare init halt)

(defmethod ig/init-key :via/service
  [_ config]
  (init config))

(defmethod ig/halt-key! :via/service
  [_ service]
  (halt service))

(declare pr-service release export compress serialize decompress deserialize)

(def ^:private future-proxy-id ":via.proxy/future")

(deftype Service [id singleton
                  token-fn encode decode exports
                  proxy-gc proxy-lock proxies
                  meta-map]
  Object
  (toString
    [^Service this]
    (pr-service this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash)
    [_]
    (hash [:via/service id exports proxy-lock]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv)
    [^Service this other]
    (boolean
     (when (instance? Service other)
       (= (#?(:clj .id :cljs .-id) this)
          (#?(:clj .id :cljs .-id) ^Service other)))))

  #?(:clj IObj :cljs IWithMeta)
  (#?(:clj withMeta :cljs -with-meta)
    [_ meta-map]
    (Service. id singleton
              token-fn encode decode exports
              proxy-gc proxy-lock proxies
              meta-map))

  IMeta
  (#?(:clj meta :cljs -meta)
    [_]
    meta-map)

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-service this)))]))

#?(:clj
   (defmethod print-method Service
     [^Service p w]
     (.write ^java.io.Writer w ^String (pr-service p))))

#?(:clj
   (defmethod simple-dispatch Service
     [^Service p]
     (print (pr-service p))))

(defonce ^:private singleton-service (atom nil))

(defn init ^Service
  [{:keys [singleton proxy-gc exports transit-handlers]
    :or {proxy-gc (td/minutes 1)}}]
  (let [read  {:handlers (merge (:read  tt/handlers)
                                (:read  transit-handlers))}
        write {:handlers (merge (:write tt/handlers)
                                (:write transit-handlers))}
        token-counter (atom (long (/ (t/into :long (t/now)) 1000)))
        token-fn #(swap! token-counter inc)
        encode (comp (partial serialize write) compress)
        decode (comp decompress (partial deserialize read))
        proxies (atom {})
        service (Service.
                 (str (uuid/random))
                 singleton
                 token-fn encode decode
                 (atom {:event #{:-vs/s :-vs/d :-vss/u}})
                 proxy-gc #?(:clj (ReentrantLock.) :cljs nil)
                 proxies
                 nil)]
    (when singleton
      (swap! proxies
             assoc future-proxy-id
             (proxy future-proxy-id
                 token-fn encode decode
                 (p/promise {:label ":via.connection/future"}))))
    (export service exports)
    (when singleton
      (reset! via.service/singleton-service service))
    service))

(defn halt
  [^Service service]
  (doseq [[_ proxy] @(#?(:clj .proxies :cljs .-proxies) service)]
    (release service proxy))
  (when (#?(:clj .singleton :cljs .-singleton) service)
    (reset! via.service/singleton-service nil)))

(declare handshake handle-message)

(defn bind
  [^Service service connection {:keys [coeffects]}]
  (log/trace [:via/bind service connection])
  (let [proxies (#?(:clj .proxies :cljs .-proxies) service)
        proxy-gc (#?(:clj .proxy-gc :cljs .-proxy-gc) service)
        proxy-lock (#?(:clj .proxy-lock :cljs .-proxy-lock) service)
        token-fn (#?(:clj .token-fn :cljs .-token-fn) service)
        encode (#?(:clj .encode :cljs .-encode) service)
        decode (#?(:clj .decode :cljs .-decode) service)]
    (-> (handshake service connection)
        (p/then
          (fn [{:keys [remote-id incoming]}]
            (try
              (with-lock [proxy-lock]
                (let [connection-p (p/resolved! connection {:label (str ":via.connection/" remote-id)})]
                  (if-let [^Proxy proxy (get @proxies remote-id)]
                    (let [connection-a (#?(:clj .connection :cljs .-connection) proxy)]
                      (when (p/realized? @connection-a)
                        (f/close! @@connection-a))
                      (reset! connection-a connection-p)
                      (log/trace [:via.proxy/reconnected (:id proxy)]))
                    (swap! proxies (fn [proxies]
                                     (if-let [^Proxy proxy (get proxies future-proxy-id)]
                                       (-> proxies
                                           (assoc remote-id
                                                  (let [current-connection (#?(:clj .connection :cljs .-connection) proxy)]
                                                    (when-not (p/realized? @current-connection)
                                                      (p/resolve! @current-connection connection))
                                                    (reset! (#?(:clj .id :cljs .-id) proxy) remote-id)
                                                    (reset! (#?(:clj .connection :cljs .-connection) proxy) connection-p)
                                                    (reset! (#?(:clj .coeffects :cljs .-coeffects) proxy) coeffects)
                                                    proxy))
                                           (dissoc future-proxy-id))
                                       (-> proxies
                                           (assoc remote-id
                                                  (proxy remote-id
                                                      token-fn encode decode
                                                      connection-p coeffects))))))))
                (let [^Proxy proxy (get @proxies remote-id)]
                  (f/on-close
                   connection
                   (fn [_]
                     (let [connection-p (p/promise)
                           connection-a (#?(:clj .connection :cljs .-connection) proxy)]
                       (log/debug [:via.proxy/disconnected (:id proxy) :gc proxy-gc])
                       (reset! connection-a connection-p)
                       (timer/run-after
                        (fn []
                          (log/trace [:via.proxy/gc :check (:id proxy) connection-a connection-p
                                      (and (not (p/realized? @connection-a))
                                           (= @connection-a connection-p))])
                          (when (and (not (p/realized? @connection-a))
                                     (= @connection-a connection-p))
                            (log/debug [:via.proxy/gc :dispose (:id proxy)])
                            (release service proxy)))
                        (td/into :milliseconds proxy-gc))
                       (reset! (#?(:clj .status :cljs .-status) proxy) :disconnected))))
                  (doseq [m incoming] (handle-message service proxy m))
                  (f/consume (fn [m] (handle-message service proxy (decode m))) connection)
                  (log/trace [:via.proxy/connected (:id proxy)])
                  (reset! (#?(:clj .status :cljs .-status) proxy) :connected)))
              (catch #?(:clj Throwable :cljs :default) e
                (log/error [:via.bind/handshake] e)))))))
  connection)

(defn bind-factory
  [^Service service connection-fn
   {:keys [on-connect on-disconnect retry-schedule]
    :or {retry-schedule (fn []
                          (ts/exponential
                           (t/now)
                           {:factor 2
                            :first-step (td/milliseconds 100)
                            :max-step (td/milliseconds 5000)}))}
    :as opts}]
  (let [running #?(:clj (ReentrantLock.) :cljs nil)
        retry
        (fn retry [schedule connect-fn]
          (when (or (not schedule) (f/closed? schedule))
            (let [retry-schedule (fs/at (retry-schedule))]
              (f/consume (fn [ts]
                           (connect-fn retry-schedule ts))
                         retry-schedule))))
        connect
        (fn connect [schedule ts]
          (when #?(:clj (.tryLock running) :cljs true)
            (log/trace [:via/connect service (or ts :initial)])
            (-> (connection-fn)
                (p/then
                  (fn [connection]
                    (when schedule (f/close! schedule))
                    (f/on-close
                     connection
                     (fn [_]
                       (log/trace [:via.connection/retry service connection])
                       (when on-disconnect
                         (on-disconnect connection))
                       (retry schedule connect)))
                    (bind service connection opts)
                    (when on-connect
                      (on-connect connection))))
                (p/finally
                  (fn []
                    #(:clj (.unlock running)))))))]
    (connect nil nil)))

(defn release
  [^Service service ^Proxy proxy]
  (log/trace [:via/release (:id proxy)])
  (let [proxies (#?(:clj .proxies :cljs .-proxies) service)]
    (reset! (#?(:clj .status :cljs .-status) proxy) :released)
    (swap! proxies dissoc (:id proxy))
    (when-let [connection-p @(#?(:clj .connection :cljs .-connection) proxy)]
      (when (p/realized? connection-p)
        (f/close! @connection-p)))))

(defn only-service
  []
  (let [^Service service @via.service/singleton-service]
    (if (#?(:clj .singleton :cljs .-singleton) service)
      @via.service/singleton-service
      (throw (ex-info ":via/service not a singleton" {})))))

(defn only-proxy
  [^Service service]
  (if (#?(:clj .singleton :cljs .-singleton) service)
    (-> @(#?(:clj .proxies :cljs .-proxies) service)
        first second)
    (throw (ex-info ":via/service not a singleton" {}))))

(defn export
  ([^Service service exports]
   (doseq [[key exports] exports]
     #?(:cljs
        (when (= key :namespaces)
          (throw
           (ex-info
            "Namespace exports are only available in Clojure."
            {:exports exports}))))
     (doseq [id exports]
       (condp = key
         :events (export service :event id)
         :subs (export service :sub id)
         :namespaces (export service :namespace id)))))
  ([^Service service type id]
   (let [exports (#?(:clj .exports :cljs .-exports) service)]
     (swap! exports update type #(conj (set %) id)))))

(defn exported?
  [^Service service type id]
  (let [exports (#?(:clj .exports :cljs .-exports) service)]
    (boolean
     (or (get-in @exports [type id])
         (when (#{:sub :event} type)
           (let [namespace (ns/normalize
                            (if (= :event type)
                              (se/namespace id)
                              (ss/namespace id)))]
             (some #(= (ns/normalize %) namespace)
                   (:namespace @exports))))))))


;;; Effect Handlers

(sfx/reg-fx
 :via/disconnect
 (fn [{:keys [service proxy]} _data]
   (release service proxy)))

(declare reply)

(sfx/reg-fx
 :via/reply
 (fn [{:keys [proxy message]} reply]
   (via.service/reply proxy message reply)))


;;; Private

(def ^:private key-encoder
  {:status :s
   :body :b
   :headers :h
   :type :t
   :reply-id :r})

(def ^:private key-decoder
  (map-invert key-encoder))

(def ^:private type-encoder
  {:event 0
   :reply 1
   :offer 2
   :answer 3})

(def ^:private type-decoder
  (map-invert type-encoder))

(defn- compress
  [message]
  (-> message
      (update :type type-encoder)
      (rename-keys key-encoder)))

(defn- decompress
  [message]
  (-> message
      (rename-keys key-decoder)
      (update :type type-decoder)))

(defn- serialize
  [handlers message]
  #?(:clj (let [out (ByteArrayOutputStream. 4096)]
            (transit-clj/write (transit-clj/writer out :json handlers) message)
            (.toString out))
     :cljs (transit-cljs/write (transit-cljs/writer :json handlers) message)))

(defn- deserialize
  [handlers ^String message]
  #?(:clj (let [in (ByteArrayInputStream. (.getBytes message))]
            (transit-clj/read (transit-clj/reader in :json handlers)))
     :cljs (transit-cljs/read (transit-cljs/reader :json handlers) message)))

(defn- handshake
  [^Service service connection]
  (let [id (#?(:clj .id :cljs .-id) service)
        encode (#?(:clj .encode :cljs .-encode) service)
        decode (#?(:clj .decode :cljs .-decode) service)
        done (p/promise {:label (str "(handshake " service " " connection ")")})
        error (fn [e]
                (when-not (p/realized? done)
                  (p/reject! done e)))
        remote-id (atom nil)
        complete (atom nil)
        incoming (atom [])]
    (-> (f/consume
         (fn [^String raw-message]
           (try
             (let [{:keys [type status body] :as m} (decode raw-message)]
               (log/trace [:via.handshake/< m])
               (case type
                 :offer (do
                          (reset! remote-id (:i body))
                          (f/put! connection (encode {:type :answer
                                                      :status :ok})))
                 :answer (reset! complete status)
                 (swap! incoming conj m))
               (when (and @remote-id @complete)
                 f/consume-stop))
             (catch #?(:clj Throwable :cljs :default) e
               (log/error [:via.handshake/error service connection raw-message] e))))
         connection)
        (p/then
          (fn [_]
            (if (and @remote-id @complete)
              (p/resolve! done {:remote-id @remote-id
                                :incoming @incoming})
              (error :closed)))) ;; TODO: error
        (p/catch error))
    (let [offer {:type :offer
                 :body {:i id
                        :l #?(:clj :clj :cljs :cljs)
                        :v (jar/version "com.7theta" "via")}}]
      (-> (f/put! connection (encode offer))
          (p/then
            (fn [_]
              (log/trace [:via.handshake/> offer])))
          (p/catch error)))
    done))

(defn- reply
  [^Proxy proxy {:keys [reply-id] :as message} reply]
  (when reply-id
    (log/trace [:via/reply (:id proxy) message reply])
    (vp/send proxy {:type :reply :reply-id reply-id :body reply})))

(defn handle-error
  [^Service service ^Proxy proxy message error]
  (log/error [:via/error service (:id proxy) message] error))

(defn- handle-reply
  [_service ^Proxy proxy {:keys [body reply-id] :as _message}]
  (let [requests (#?(:clj .requests :cljs .-requests) proxy)
        {:keys [on-reply timer]} (get @requests reply-id)]
    (swap! requests dissoc reply-id)
    (when timer (timer/cancel timer))
    (when on-reply (on-reply body))))

(defn- handle-event
  [^Service service ^Proxy proxy {:keys [body] :as message}]
  (let [[event-id & _] body
        coeffects (#?(:clj .coeffects :cljs .-coeffects) proxy)]
    (if (and (se/event? event-id)
             (exported? service :event event-id))
      (se/dispatch (merge @coeffects
                          {:message message
                           :proxy proxy
                           :service service})
                   body)
      (let [error-text (str ":via/event :unknown " event-id)]
        (reply proxy message {:status 400
                              :body error-text})
        (handle-error service proxy message (ex-info error-text {:proxy proxy :message message}))))))

(defn- handle-message
  [^Service service ^Proxy proxy {:keys [type] :as message}]
  (let [id (:id proxy)]
    (try
      (log/trace [:via/< id message])
      (case type
        :event (handle-event service proxy message)
        :reply (handle-reply service proxy message)
        (handle-error service proxy message (ex-info "via/message :type :unknown" {:message message})))
      (catch #?(:clj Throwable :cljs :default) e
        (handle-error service proxy message e)))))

(defn- pr-service
  [^Service s]
  (ust/format
   (str "#<via/service@" #?(:clj "0x%x" :cljs "%s") "%s%s>")
   (hash s)
   (str "[" (if (#?(:clj .singleton
                    :cljs .-singleton) s)
              "singleton"
              (#?(:clj .id :cljs .-id) s)) "] ")
   (str (vec (vals @(#?(:clj .proxies :cljs .-proxies) s))))))
