;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.re-frame
  (:require [via.endpoint :as via]
            [via.defaults :as defaults]
            [via.adapter :as adapter]
            [via.subs :as vs]
            [via.events :as ve]
            [via.adapter :as va]
            [signum.fx :as sfx]
            [reagent.ratom :as ra]
            [re-frame.core :as rf]
            [re-frame.registrar :as rfr]
            [re-frame.subs :as rfs]
            [utilis.js :as j]
            [clojure.data :refer [diff]]))

(declare path subscriptions re-frame-handlers)

(defn subscribe
  [endpoint peer-id [query-id & _ :as query-v] default]
  (let [subscriptions (subscriptions endpoint peer-id)]
    (when-not (re-frame.registrar/get-handler :sub query-id)
      (rf/reg-sub
       query-id
       (fn [db query-v]
         (let [path (path query-v)]
           (swap! subscriptions conj query-v)
           (if (contains? (get-in db (drop-last path)) (last path))
             (get-in db path)
             default)))))
    (ra/make-reaction (fn [] @(rf/subscribe query-v)))))

(defn dispatch
  [endpoint peer-id event]
  (if (re-frame.registrar/get-handler :event (first event))
    (rf/dispatch event)
    (ve/dispatch endpoint peer-id event)))

(defn invoke
  [endpoint peer-id event options]
  (ve/invoke endpoint peer-id event (re-frame-handlers options)))

;;; Effect Handlers

(rf/reg-fx
 :via/dispatch
 (fn [event-or-map]
   ((-> @sfx/effect-handlers
        :via/dispatch
        :handler) nil event-or-map)))

(rf/reg-fx
 :via/invoke
 (fn [{:keys [event
             timeout
             on-success
             on-failure
             on-timeout
             endpoint
             peer-id]
      :as options}]
   (-> ((-> @sfx/effect-handlers
            :via/invoke
            :handler) nil options)
       (j/call :then (fn [_]))
       (j/call :catch (fn [_])))))

;;; Event Handlers

(rf/reg-event-fx
 :via/dispatch
 (fn [_ request-map-or-seq]
   {:via/dispatch request-map-or-seq}))

(rf/reg-event-fx
 :via.session-context/replace
 (fn [_ session-context]
   {:via/dispatch {:event [:via.session-context/replace {:session-context session-context
                                                         :sync false}]}}))

(rf/reg-event-fx
 :via.session-context/merge
 (fn [_ session-context]
   {:via/dispatch {:event [:via.session-context/merge {:session-context session-context
                                                       :sync false}]}}))

;;; Implementation

(defn- path
  [query-v]
  [:via.subs/cache query-v])

(rf/reg-event-fx
 :via.re-frame.sub.value/updated
 (fn [{:keys [db]} [_ query-v value]]
   {:db (assoc-in db (path query-v) value)}))

(rf/reg-event-fx
 :via.re-frame.sub.value/removed
 (fn [{:keys [db]} [_ query-v]]
   (let [path (path query-v)]
     {:db (update-in db (drop-last path) dissoc (last path))})))

(defn remote-subscribe
  [endpoint peer-id remote-subscriptions query-v]
  (let [sub (vs/subscribe endpoint peer-id query-v)]
    (add-watch sub ::remote-subscribe
               (fn [_ _ _ value]
                 (rf/dispatch [:via.re-frame.sub.value/updated query-v value])))
    (swap! remote-subscriptions assoc query-v sub)
    sub))

(defn remote-dispose
  [endpoint peer-id remote-subscriptions query-v]
  (when-let [sub (get @remote-subscriptions query-v)]
    (rf/dispatch [:via.re-frame.sub.value/removed query-v])
    (remove-watch sub ::remote-subscribe)
    (swap! remote-subscriptions dissoc query-v)
    (vs/dispose endpoint peer-id query-v)))

(defn subscriptions
  [endpoint peer-id]
  (let [context (va/context (endpoint))]
    (or (get-in @context [peer-id ::subscriptions])
        (let [subscriptions (atom #{})
              remote-subscriptions (atom {})]
          (swap! context assoc-in [peer-id ::subscriptions] subscriptions)
          (add-watch rfs/query->reaction :via.re-frame/subscription-cache
                     (fn [_key _ref old-value new-value]
                       (reset! subscriptions
                               (->> new-value
                                    keys
                                    (map first)
                                    (filter @subscriptions)
                                    set))))
          (add-watch subscriptions ::subscriptions
                     (fn [_key _ref old-value new-value]
                       (let [[removed added _] (diff old-value new-value)]
                         (doseq [query-v removed]
                           (when-not (->> @subscriptions
                                          (filter #(= (first %) (first query-v)))
                                          not-empty)
                             (rfr/clear-handlers :sub (first query-v)))
                           (remote-dispose endpoint peer-id remote-subscriptions query-v))
                         (doseq [query-v added]
                           (remote-subscribe endpoint peer-id remote-subscriptions query-v)))))
          subscriptions))))

(defn re-frame-handlers
  [options]
  (reduce (fn [options key]
            (if-let [handler (get options key)]
              (assoc options key
                     (if (vector? handler)
                       (fn [& args]
                         (rf/dispatch
                          (vec (concat handler args))))
                       handler))
              options))
          options
          [:on-success
           :on-failure
           :on-timeout]))
