(ns coconut.v1.documentation
  #?(:cljs (:require-macros [cljs.core.async.macros :as async]))
  (:require
    [coconut.v1.platform :as platform]
    [coconut.v1.summarizing :as summarizing]
    [coconut.v1.rendering :as rendering]
    [clojure.core.async :as async :refer [<! >!]]
    [clojure.pprint :as cp]
    [clojure.string :as cs]
    ))

(defn ^{:private true} with-indicies
  ([elements]
   (into (vector)
         (comp (map-indexed vector)
               (map #(update % 0 inc)))
         elements)))

(def ^{:private true :const true} checkmark \✔)
(def ^{:private true :const true} x \✘)
(def ^{:private true :const true} asterisk \*)

(defmulti ^{:private true} report-event
  (fn [state event output-channel]
    (::summarizing/type event)))

(defmethod ^{:private true} report-event
  ::summarizing/suite-started
  ([state event output-channel]
   (async/go
     (merge state
            #::{:total-number-of-tests (::summarizing/total-number-of-tests event)}))))

(defmethod ^{:private true} report-event
  ::summarizing/context-started
  ([state event output-channel]
   (async/go
     (>! output-channel
         #::rendering{:type ::rendering/line
                      :segments [#::rendering{:text (rendering/indent (::level state))}
                                 #::rendering{:text (rendering/render-subject (::summarizing/subject event))}]})
     (update state
             ::level inc))))

(defmethod ^{:private true} report-event
  ::summarizing/context-finished
  ([state event output-channel]
   (async/go
     (update state
             ::level dec))))

(defmethod ^{:private true} report-event
  ::summarizing/test-pending
  ([state event output-channel]
   (async/go
     (>! output-channel
         #::rendering{:type ::rendering/line
                      :segments [#::rendering{:text (rendering/indent (::level state))}
                                 #::rendering{:text asterisk
                                              :color ::rendering/yellow}
                                 #::rendering{:text (rendering/spacing 1)}
                                 #::rendering{:text (::summarizing/description event)
                                              :color ::rendering/yellow}]})
     (update state
             ::pending-tests conj event))))

(defmethod ^{:private true} report-event
  ::summarizing/test-passed
  ([state event output-channel]
   (async/go
     (>! output-channel
         #::rendering{:type ::rendering/line
                      :segments [#::rendering{:text (rendering/indent (::level state))}
                                 #::rendering{:text checkmark
                                              :color ::rendering/green}
                                 #::rendering{:text (rendering/spacing 1)}
                                 #::rendering{:text (::summarizing/description event)
                                              :color ::rendering/green}]})
     state)))

(defmethod ^{:private true} report-event
  ::summarizing/test-failed
  ([state event output-channel]
   (async/go
     (>! output-channel
         #::rendering{:type ::rendering/line
                      :segments [#::rendering{:text (rendering/indent (::level state))}
                                 #::rendering{:text x
                                              :color ::rendering/red}
                                 #::rendering{:text (rendering/spacing 1)}
                                 #::rendering{:text (::summarizing/description event)
                                              :color ::rendering/red}]})
     (update state
             ::failing-tests conj event))))

(defmethod ^{:private true} report-event
  ::summarizing/test-threw-exception
  ([state event output-channel]
   (async/go
     (>! output-channel
         #::rendering{:type ::rendering/line
                      :segments [#::rendering{:text (rendering/indent (::level state))}
                                 #::rendering{:text x
                                              :color ::rendering/red}
                                 #::rendering{:text (rendering/spacing 1)}
                                 #::rendering{:text (::summarizing/description event)
                                              :color ::rendering/red}]})
     (update state
             ::failing-tests conj event))))

(defmethod ^{:private true} report-event
  ::summarizing/test-timed-out
  ([state event output-channel]
   (async/go
     (>! output-channel
         #::rendering{:type ::rendering/line
                      :segments [#::rendering{:text (rendering/indent (::level state))}
                                 #::rendering{:text x
                                              :color ::rendering/red}
                                 #::rendering{:text (rendering/spacing 1)}
                                 #::rendering{:text (::summarizing/description event)
                                              :color ::rendering/red}]})
     (update state
             ::failing-tests conj event))))

(defmethod ^{:private true} report-event
  ::summarizing/suite-finished
  ([state event output-channel]
   (async/go
     (when (seq (::pending-tests state))
       (>! output-channel
           #::rendering{:type ::rendering/newline})
       (>! output-channel
           #::rendering{:type ::rendering/line
                        :segments [#::rendering{:text "pending..."}]})
       (doseq [[n t] (with-indicies (::pending-tests state))]
         (let [list-index (rendering/render-list-index n)]
           (>! output-channel
               #::rendering{:type ::rendering/newline})
           (>! output-channel
               #::rendering{:type ::rendering/line
                            :segments [#::rendering{:text (rendering/indent 1)}
                                       #::rendering{:text list-index}
                                       #::rendering{:text (rendering/spacing 1)}
                                       #::rendering{:text (rendering/render-full-description (::summarizing/context t)
                                                                                             (::summarizing/description t))}]})
           (>! output-channel
               #::rendering{:type ::rendering/line
                            :segments [#::rendering{:text (rendering/indent 1)}
                                       #::rendering{:text (rendering/spacing (count list-index))}
                                       #::rendering{:text (rendering/spacing 1)}
                                       #::rendering{:text (rendering/pending-reason (::summarizing/pending-reason t))
                                                    :color ::rendering/yellow}]}))))

     (when (seq (::failing-tests state))
       (>! output-channel
           #::rendering{:type ::rendering/newline})
       (>! output-channel
           #::rendering{:type ::rendering/line
                        :segments [#::rendering{:text "failures..."}]})
       (doseq [[n t] (with-indicies (::failing-tests state))]
         (let [list-index (rendering/render-list-index n)]
           (>! output-channel
               #::rendering{:type ::rendering/newline})
           (>! output-channel
               #::rendering{:type ::rendering/line
                            :segments [#::rendering{:text (rendering/indent 1)}
                                       #::rendering{:text list-index}
                                       #::rendering{:text (rendering/spacing 1)}
                                       #::rendering{:text (rendering/render-full-description (::summarizing/context t)
                                                                                             (::summarizing/description t))}]})
           (case (::summarizing/type t)
             ::summarizing/test-failed
             (do (>! output-channel
                     #::rendering{:type ::rendering/line
                                  :segments [#::rendering{:text (rendering/indent 1)}
                                             #::rendering{:text (rendering/spacing (count list-index))}
                                             #::rendering{:text (rendering/spacing 3)}
                                             #::rendering{:text "expected:"
                                                          :color ::rendering/teal}
                                             #::rendering{:text (rendering/spacing 1)}
                                             #::rendering{:text (::summarizing/expected t)
                                                          :color ::rendering/red}]})
                 (>! output-channel
                     #::rendering{:type ::rendering/line
                                  :segments [#::rendering{:text (rendering/indent 1)}
                                             #::rendering{:text (rendering/spacing (count list-index))}
                                             #::rendering{:text (rendering/spacing 8)}
                                             #::rendering{:text "got:"
                                                          :color ::rendering/teal}
                                             #::rendering{:text (rendering/spacing 1)}
                                             #::rendering{:text (::summarizing/actual t)
                                                          :color ::rendering/red}]}))

             ::summarizing/test-threw-exception
             (let [e (::summarizing/exception t)
                   stack-trace (platform/exception-stack-trace e)]
               (>! output-channel
                   #::rendering{:type ::rendering/line
                                :segments [#::rendering{:text (rendering/indent 1)}
                                           #::rendering{:text (rendering/spacing (count list-index))}
                                           #::rendering{:text (rendering/spacing 2)}
                                           #::rendering{:text "exception:"
                                                        :color ::rendering/teal}
                                           #::rendering{:text (rendering/spacing 1)}
                                           #::rendering{:text (platform/exception-class-name e)
                                                        :color ::rendering/red}]})
               (>! output-channel
                   #::rendering{:type ::rendering/line
                                :segments [#::rendering{:text (rendering/indent 1)}
                                           #::rendering{:text (rendering/spacing (count list-index))}
                                           #::rendering{:text (rendering/spacing 4)}
                                           #::rendering{:text "message:"
                                                        :color ::rendering/teal}
                                           #::rendering{:text (rendering/spacing 1)}
                                           #::rendering{:text (or (platform/exception-message e) "-")
                                                        :color ::rendering/red}]})
               (>! output-channel
                   #::rendering{:type ::rendering/line
                                :segments [#::rendering{:text (rendering/indent 1)}
                                           #::rendering{:text (rendering/spacing (count list-index))}
                                           #::rendering{:text "stack trace:"
                                                        :color ::rendering/teal}
                                           #::rendering{:text (rendering/spacing 1)}
                                           #::rendering{:text (first stack-trace)
                                                        :color (rendering/stack-trace-element-color (first stack-trace))}]})
               (doseq [stack-trace-element (rest stack-trace)]
                 (>! output-channel
                     #::rendering{:type ::rendering/line
                                  :segments [#::rendering{:text (rendering/indent 1)}
                                             #::rendering{:text (rendering/spacing (count list-index))}
                                             #::rendering{:text (rendering/spacing 13)}
                                             #::rendering{:text stack-trace-element
                                                          :color (rendering/stack-trace-element-color stack-trace-element)}]})))

             ::summarizing/test-timed-out
             (do (>! output-channel
                     #::rendering{:type ::rendering/line
                                  :segments [#::rendering{:text (rendering/indent 1)}
                                             #::rendering{:text (rendering/spacing (count list-index))}
                                             #::rendering{:text (rendering/spacing 5)}
                                             #::rendering{:text "reason:"
                                                          :color ::rendering/teal}
                                             #::rendering{:text (rendering/spacing 1)}
                                             #::rendering{:text "TIMEOUT"
                                                          :color ::rendering/red}]}))

             (throw (platform/illegal-argument-exception
                      (str "unsupported summarizing event type: "
                           (::summarizing/type t)))))

           (>! output-channel
               #::rendering{:type ::rendering/newline})
           (>! output-channel
               #::rendering{:type ::rendering/line
                            :segments [#::rendering{:text (rendering/indent 1)}
                                       #::rendering{:text (rendering/spacing (count list-index))}
                                       #::rendering{:text (rendering/spacing 2)}
                                       #::rendering{:text "namespace:"
                                                    :color ::rendering/teal}
                                       #::rendering{:text (rendering/spacing 1)}
                                       #::rendering{:text (::summarizing/namespace-name t)
                                                    :color ::rendering/red}]})
           (>! output-channel
               #::rendering{:type ::rendering/line
                            :segments [
                                       #::rendering{:text (rendering/indent 1)}
                                       #::rendering{:text (rendering/spacing (count list-index))}
                                       #::rendering{:text (rendering/spacing 7)}
                                       #::rendering{:text "line:"
                                                    :color ::rendering/teal}
                                       #::rendering{:text (rendering/spacing 1)}
                                       #::rendering{:text (str (::summarizing/definition-line-number t))
                                                    :color ::rendering/red}]}))))

     (>! output-channel
         #::rendering{:type ::rendering/newline})
     (>! output-channel
         #::rendering{:type ::rendering/line
                      :segments [#::rendering{:text (platform/format "finished in %s seconds"
                                                                     (rendering/render-suite-duration (::summarizing/duration-in-milliseconds event)))}]})
     (>! output-channel
         #::rendering{:type ::rendering/line
                      :segments [#::rendering{:text (rendering/render-number-of-tests (::total-number-of-tests state)
                                                                                      (count (::failing-tests state))
                                                                                      (count (::pending-tests state)))
                                              :color (rendering/number-of-tests-color (count (::failing-tests state))
                                                                                      (count (::pending-tests state)))}]})
     (async/close! output-channel))))

(defmethod ^{:private true} report-event
  :default
  ([state event output-channel]
   (async/go state)))

(defn report
  "Given a core.async channel of summarized test events, returns a new channel
  containing data structures describing the information which should be printed
  to the screen."
  ([event-channel]
   (let [output-channel (async/chan 250)]
     (async/go
       (loop [state #::{:total-number-of-tests nil
                        :pending-tests []
                        :failing-tests []
                        :level 0}]
         (when-let [event (<! event-channel)]
           (let [updated-state (<! (report-event state event output-channel))]
             (recur updated-state))))
       (async/close! output-channel))
     output-channel)))
