(ns coconut.v1.query
  (:refer-clojure :exclude [and or not])
  (:require
    [coconut.v1.platform :as platform]
    [coconut.v1.core :as core]
    [clojure.set :as cs]
    ))

(defn ^{:private true} satisfied?
  ([criteria state]
   ((::predicate criteria) state)))

(defmulti ^{:private true} number-of-tests
  (fn [component]
    (::core/component-type component)))

(defmethod ^{:private true} number-of-tests
  ::result
  ([component]
   (number-of-tests (::core/sub-components component))))

(defmethod ^{:private true} number-of-tests
  ::core/test
  ([component] 1))

(defmethod ^{:private true} number-of-tests
  ::core/context
  ([component]
   (number-of-tests (::core/sub-components component))))

(defmethod ^{:private true} number-of-tests
  ::core/collection
  ([component]
   (transduce (map number-of-tests)
              +
              (::core/components component))))

(defn ^{:private true} with-current-component-type
  ([state component]
   (assoc state
          ::current-component-type (::core/component-type component))))

(defmulti ^{:private true} with-merged-description
  (fn [state component]
    (::description state)))

(defmethod ^{:private true} with-merged-description
  ""
  ([state component]
   (assoc state ::description (core/string-label component))))

(defmethod ^{:private true} with-merged-description
  :default
  ([state component]
   (update state ::description str \space (core/string-label component))))

(defn ^{:private true} with-namespace-name
  ([state component]
   (assoc state
          ::namespace-name (::core/namespace-name component))))

(defn ^{:private true} with-definition-line-number
  ([state component]
   (assoc state
          ::definition-line-number (::core/definition-line-number component))))

(defn ^{:private true} with-tags
  ([state component]
   (update state
           ::tags (partial cs/union
                           (set (::core/tags component))))))

(defn ^{:private true} merge-component-information
  ([state component]
   (-> state
       (with-current-component-type component)
       (with-merged-description component)
       (with-namespace-name component)
       (with-definition-line-number component)
       (with-tags component))))

(declare ^{:private true} filter-components)

(defmulti ^{:private true} filter-component
  (fn [criteria state component]
    (::core/component-type component)))

(defmethod ^{:private true} filter-component
  ::core/collection
  ([criteria state component]
   (update component
           ::core/components
           (partial into
                    (vector)
                    (filter-components criteria state)))))

(defmethod ^{:private true} filter-component
  ::core/context
  ([criteria state component]
   (let [updated-state (merge-component-information state component)]
     (if (satisfied? criteria updated-state)
       component
       (update component
               ::core/sub-components
               (partial filter-component
                        criteria
                        updated-state))))))

(defmethod ^{:private true} filter-component
  ::core/test
  ([criteria state component]
   (let [updated-state (merge-component-information state component)]
     (when (satisfied? criteria updated-state)
       component))))

(defn ^{:private true} filter-components
  ([criteria state]
   (comp (map (partial filter-component criteria state))
         (remove nil?)
         (remove (comp zero? number-of-tests)))))

(defn ^{:private true} build-query-result
  ([]
   #::core{:component-type ::result
           ::total-number-of-tests 0
           :sub-components (core/create-collection-component)})
  ([result component]
   (-> result
       (update ::total-number-of-tests + (number-of-tests component))
       (update ::core/sub-components core/append-component-to-collection component))))

(defn ^{:private true} or*
  ([criteria]
   (fn [state]
     (loop [criteria criteria]
       (if criteria
         (let [criterion (first criteria)]
           (if (satisfied? criterion state)
             true
             (recur (next criteria))))
         false)))))

(defn ^{:private true} and*
  ([criteria]
   (fn [state]
     (loop [criteria criteria]
       (if criteria
         (let [criterion (first criteria)]
           (if (satisfied? criterion state)
             (recur (next criteria))
             false))
         true)))))

(defn ^{:export true} not
  "Negates the given criteria."
  ([criteria]
   {::predicate
    (::negated-predicate criteria (complement (::predicate criteria)))
    ::negated-predicate
    (::predicate criteria)}))

(defn ^{:export true} or
  "Combines two criteria with logical OR."
  ([criterion & criteria]
   (let [criteria (list* criterion criteria)]
     {::predicate
      (or* criteria)
      ::negated-predicate
      (and* (sequence (map not) criteria))})))

(defn ^{:export true} and
  "Combines two criteria with logical AND."
  ([criterion & criteria]
   (let [criteria (list* criterion criteria)]
     {::predicate
      (and* criteria)
      ::negated-predicate
      (or* (sequence (map not) criteria))})))

(def ^{:export true} all
  "Returns all defined tests."
  {::predicate
   (constantly true)})

(defn ^{:export true} within-namespace
  "Returns all tests which are defined in the given namespace."
  ([namespace-name]
   {::predicate
    (fn [state]
      (= (str namespace-name)
         (::namespace-name state)))}))

(defn ^{:export true} within-namespaces
  "Returns all tests which are defined in the given namespaces."
  ([namespace-name & namespace-names]
   (transduce (map within-namespace)
              (completing or)
              (not all)
              (list* namespace-name namespace-names))))

(defn ^{:export true} namespace-matches
  "Returns all tests whose namespace name maches the given regex."
  ([regex]
   {::predicate
    (fn [state]
      (boolean (re-matches regex
                           (::namespace-name state))))}))

(defn ^{:export true} defined-on-line
  "Returns all tests which are defined on the given line. If a context is
  defined on the given line all tests within it will be returned."
  ([line-number]
   {::predicate
    (fn [state]
      (= line-number
         (::definition-line-number state)))
    ::negated-predicate
    (fn [state]
      (clojure.core/and (= ::core/test
                           (::current-component-type state))
                        (not= line-number
                              (::definition-line-number state))))}))

(defn ^{:export true} description-matches
  "Returns all tests whose full description matches the given regex. The
  full description is calculating by combining all context subjects with
  the test description."
  ([regex]
   {::predicate
    (fn [state]
      (clojure.core/and (= ::core/test
                           (::current-component-type state))
                        (boolean (re-matches regex
                                             (::description state)))))
    ::negated-predicate
    (fn [state]
      (clojure.core/and (= ::core/test
                           (::current-component-type state))
                        (clojure.core/not (boolean (re-matches regex
                                                               (::description state))))))}))

(defn ^{:export true} has-tag
  "Returns all tests ore contexts which are tagged with the given value."
  ([tag]
   {::predicate
    (fn [state]
      (contains? (::tags state) tag))}))

(defn query
  "Given a predicate which takes a test component, returns a result
  containing all tests for which the predicate is satisfied. If a context
  is empty or no tests within a context satisfy the predicate the context
  will not be returned."
  ([criteria]
   (query criteria
          (platform/registered-components #::platform{:component-version 1})))
  ([criteria defined-components]
   (let [state #::{:description ""
                   :definition-line-number nil
                   :namespace-name nil
                   :tags #{}}]
     (transduce (filter-components criteria state)
                (completing build-query-result)
                defined-components))))

(defn return-all
  "Given a collection of components, returns a query result
  containing those components. Does not return empty contexts."
  ([collection]
   (query all collection)))

(defn return
  "Given a component, returns a query result containing
  that component. Does not return empty contexts."
  ([component]
   (return-all [component])))
