(ns coconut.v1.query-test
  (:require
    [coconut.v1.core :as c]
    [coconut.v1.platform :as platform]
    [coconut.v1.query :as query]
    [coconut.v1.test-namespaces.a]
    [coconut.v1.test-namespaces.b]
    [coconut.v1.test-namespaces.c]
    ))

(c/describe `query/query
  (c/it "returns a query result"
    (fn [assert-that]
      (let [criteria query/all
            result (query/query criteria)]
        (assert-that result
                     (c/contains {::c/component-type ::query/result})))))

  (for [fixture [{::it "returns zero for the total number of tests when no tests are defined"
                  ::total-number-of-tests 0
                  ::defined-components []}

                 {::it "returns one for the total number of tests when one top-level test is defined"
                  ::total-number-of-tests 1
                  ::defined-components [(c/it "is a single top-level test"
                                          (fn [_]))]}

                 {::it "returns N for the total number of tests when N top-level tests are defined"
                  ::total-number-of-tests 5
                  ::defined-components [(c/it "one" (fn [_]))
                                        (c/it "two" (fn [_]))
                                        (c/it "three" (fn [_]))
                                        (c/it "four" (fn [_]))
                                        (c/it "five" (fn [_]))]}

                 {::it "returns zero when a single empty context is defined"
                  ::total-number-of-tests 0
                  ::defined-components [(c/context "empty context")]}

                 {::it "returns zero when multiple empty contexts are defined"
                  ::total-number-of-tests 0
                  ::defined-components [(c/context "one")
                                        (c/context "two")
                                        (c/context "three")]}

                 {::it "counts each test defined within a context"
                  ::total-number-of-tests 3
                  ::defined-components [(c/context "one"
                                          (c/it "one" (fn [_])))
                                        (c/context "two"
                                          (c/it "two" (fn [_])))
                                        (c/context "three"
                                          (c/it "three" (fn [_])))]}

                 {::it "counts each test within nested contexts"
                  ::total-number-of-tests 3
                  ::defined-components [(c/context "a"
                                          (c/it "one" (fn [_])))
                                        (c/context "a"
                                          (c/context "b"
                                            (c/it "two" (fn [_]))))
                                        (c/context "a"
                                          (c/context "b"
                                            (c/context "c"
                                              (c/it "three" (fn [_])))))]}

                 {::it "counts each test within nested collection components"
                  ::total-number-of-tests 3
                  ::defined-components [(c/context "a"
                                          (c/create-collection-component [(c/it "one" (fn [_]))
                                                                          (c/it "two" (fn [_]))
                                                                          (c/it "three" (fn [_]))]))]}]]
    (c/it (::it fixture)
      (fn [assert-that]
        (let [defined-components (::defined-components fixture)
              criteria query/all
              result (query/query criteria defined-components)]
          (assert-that (::query/total-number-of-tests result)
                       (c/is (::total-number-of-tests fixture))))))))

(defn create-test
  ([]
   (create-test {}))
  ([attributes]
   (c/create-test-component
     (::file-name attributes (platform/generate-uuid))
     (::namespace-name attributes (platform/generate-uuid))
     (::definition-line-number attributes (rand-nth (take 1000 (range))))
     (::description attributes (platform/generate-uuid))
     (::function attributes (fn [_])))))

(defn create-context
  ([]
   (create-context {}))
  ([attributes]
   (c/create-context-component
     (::file-name attributes (platform/generate-uuid))
     (::namespace-name attributes (platform/generate-uuid))
     (::definition-line-number attributes (rand-nth (take 1000 (range))))
     (::subject attributes (platform/generate-uuid))
     (::components attributes []))))

(defn create-collection
  ([]
   (create-collection {}))
  ([attributes]
   (c/create-collection-component
     (::components attributes []))))

(defn returned-components
  ([result]
   (::c/components (::c/sub-components result))))

(c/describe `query/all
  (c/let [criteria query/all]

    (c/it "returns no tests when none are defined"
      (fn [assert-that]
        (let [defined-components []
              result (query/query criteria defined-components)]
          (assert-that (returned-components result)
                       (c/is [])))))

    (c/it "returns all top-level tests"
      (fn [assert-that]
        (let [a (create-test)
              b (create-test)
              c (create-test)
              defined-components [a b c]
              result (query/query criteria defined-components)]
          (assert-that (returned-components result)
                       (c/is [a b c])))))

    (c/it "returns no empty top-level contexts"
      (fn [assert-that]
        (let [a (create-context)
              b (create-context)
              c (create-context)
              defined-components [a b c]
              result (query/query criteria defined-components)]
          (assert-that (returned-components result)
                       (c/is [])))))

    (c/it "returns no empty top-level collections"
      (fn [assert-that]
        (let [a (create-collection)
              b (create-collection)
              c (create-collection)
              defined-components [a b c]
              result (query/query criteria defined-components)]
          (assert-that (returned-components result)
                       (c/is [])))))

    (c/it "returns all non-empty collections"
      (fn [assert-that]
        (let [a (create-collection {::components [(create-test)]})
              b (create-collection {::components [(create-test)]})
              c (create-collection {::components [(create-test)]})
              defined-components [a b c]
              result (query/query criteria defined-components)]
          (assert-that (returned-components result)
                       (c/is [a b c])))))

    (c/it "returns all non-empty contexts"
      (fn [assert-that]
        (let [a (create-context {::components [(create-test)]})
              b (create-context {::components [(create-test)]})
              c (create-context {::components [(create-test)]})
              defined-components [a b c]
              result (query/query criteria defined-components)]
          (assert-that (returned-components result)
                       (c/is [a b c])))))

    ; (c/it "returns all defined tests from the platform-specific registry"
    ;   {:pending "until query returns empty contexts"}
    ;   (fn [assert-that]
    ;     (let [result (query/query criteria)]
    ;       (assert-that (returned-components result)
    ;                    (c/is (platform/registered-components #::platform{:component-version 1}))))))
    ))

(c/describe `query/within-namespace
  (for [{:keys [argument-type coerce]} [{:argument-type "string" :coerce str}
                                        {:argument-type "symbol" :coerce symbol}
                                        #?(:clj {:argument-type "namespace object" :coerce the-ns})]]
    (c/context (platform/format "when the namespace name argument is a %s" argument-type)
      (c/it "returns all top-level tests whose namespace name matches the value"
        (fn [assert-that]
          (let [criteria (query/within-namespace (coerce 'coconut.v1.test-namespaces.a))
                a (create-test {::namespace-name "coconut.v1.test-namespaces.a"})
                b (create-test {::namespace-name "coconut.v1.test-namespaces.b"})
                c (create-test {::namespace-name "coconut.v1.test-namespaces.c"})
                defined-components [a b c]
                result (query/query criteria defined-components)]
            (assert-that (returned-components result)
                         (c/is [a])))))

      (c/it "returns no empty top-level contexts"
        (fn [assert-that]
          (let [criteria (query/within-namespace (coerce 'coconut.v1.test-namespaces.a))
                a (create-context {::namespace-name "coconut.v1.test-namespaces.a"})
                b (create-context {::namespace-name "coconut.v1.test-namespaces.b"})
                c (create-context {::namespace-name "coconut.v1.test-namespaces.c"})
                defined-components [a b c]
                result (query/query criteria defined-components)]
            (assert-that (returned-components result)
                         (c/is [])))))

      (c/it "returns all non-empty top-level contexts"
        (fn [assert-that]
          (let [criteria (query/within-namespace (coerce 'coconut.v1.test-namespaces.a))
                a (create-context {::namespace-name "coconut.v1.test-namespaces.a"
                                   ::components [(create-test {::namespace-name "coconut.v1.test-namespaces.a"})]})
                b (create-context {::namespace-name "coconut.v1.test-namespaces.b"
                                   ::components [(create-test {::namespace-name "coconut.v1.test-namespaces.b"})]})
                c (create-context {::namespace-name "coconut.v1.test-namespaces.c"
                                   ::components [(create-test {::namespace-name "coconut.v1.test-namespaces.c"})]})
                defined-components [a b c]
                result (query/query criteria defined-components)]
            (assert-that (returned-components result)
                         (c/is [a]))))))))

(c/describe `query/within-namespaces
  (for [{:keys [argument-type coerce]} [{:argument-type "strings" :coerce str}
                                        {:argument-type "symbols" :coerce symbol}
                                        #?(:clj {:argument-type "namespace objects" :coerce the-ns})]]
    (c/context (platform/format "when the namespace name arguments are %s" argument-type)
      (c/it "returns all top-level tests whose namespace name matches any one of the given namespace names"
        (fn [assert-that]
          (let [criteria (query/within-namespaces
                           (coerce 'coconut.v1.test-namespaces.a)
                           (coerce 'coconut.v1.test-namespaces.b))
                a (create-test {::namespace-name "coconut.v1.test-namespaces.a"})
                b (create-test {::namespace-name "coconut.v1.test-namespaces.b"})
                c (create-test {::namespace-name "coconut.v1.test-namespaces.c"})
                defined-components [a b c]
                result (query/query criteria defined-components)]
            (assert-that (returned-components result)
                         (c/is [a b])))))

      (c/it "does not return empty contexts"
        (fn [assert-that]
          (let [criteria (query/within-namespaces
                           (coerce 'coconut.v1.test-namespaces.a)
                           (coerce 'coconut.v1.test-namespaces.b))
                a (create-context {::namespace-name "coconut.v1.test-namespaces.a"})
                b (create-context {::namespace-name "coconut.v1.test-namespaces.b"})
                defined-components [a b]
                result (query/query criteria defined-components)]
            (assert-that (returned-components result)
                         (c/is [])))))

      (c/it "returns all nested tests which are defined in any of the given namespaces"
        (fn [assert-that]
          (let [criteria (query/within-namespaces
                           (coerce 'coconut.v1.test-namespaces.a)
                           (coerce 'coconut.v1.test-namespaces.b))
                a (create-context {::namespace-name "coconut.v1.test-namespaces.a"
                                   ::components [(create-test {::namespace-name "coconut.v1.test-namespaces.a"})]})
                b (create-context {::namespace-name "coconut.v1.test-namespaces.b"
                                   ::components [(create-test {::namespace-name "coconut.v1.test-namespaces.b"})]})
                defined-components [a b]
                result (query/query criteria defined-components)]
            (assert-that (returned-components result)
                         (c/is [a b]))))))))

(c/describe `query/or
  (c/it "returns all tests which match either predicate"
    (fn [assert-that]
      (let [criteria (-> (query/within-namespace "a")
                         (query/or (query/within-namespaces "b")))
            a (create-test {::namespace-name "a"})
            b (create-context {::namespace-name "b"
                               ::components [(create-test {::namespace-name "b"})]})
            c (create-test {::namespace-name "c"})
            d (create-context {::namespace-name "d"
                               ::components [(create-test {::namespace-name "d"})]})
            defined-components [a b c d]
            result (query/query criteria defined-components)]
        (assert-that (returned-components result)
                     (c/is [a b]))))))

(c/describe `query/and
  (c/it "returns all tests which satisfy both predicates"
    (fn [assert-that]
      (let [criteria (-> (query/within-namespace "a")
                         (query/and (query/defined-on-line 42)))
            a1 (create-test {::namespace-name "a"
                             ::definition-line-number 42})
            a2 (create-context {::namespace-name "a"
                                ::definition-line-number 43
                                ::components [(create-test {::namespace-name "a"
                                                            ::definition-line-number 44})]})
            b1 (create-test {::namespace-name "b"
                             ::definition-line-number 42})
            b2 (create-context {::namespace-name "b"
                                ::definition-line-number 43
                                ::components [(create-test {::namespace-name "b"
                                                            ::definition-line-number 44})]})
            defined-components [a1 a2 b1 b2]
            result (query/query criteria defined-components)]
        (assert-that (returned-components result)
                     (c/is [a1]))))))

(c/describe `query/defined-on-line
  (c/it "returns all top-level tests which are defined on the given line number"
    (fn [assert-that]
      (let [criteria (query/defined-on-line 42)
            a (create-test {::definition-line-number 42})
            b (create-test {::definition-line-number 0})
            defined-components [a b]
            result (query/query criteria defined-components)]
        (assert-that (returned-components result)
                     (c/is [a])))))

  (c/it "returns all tests for a context when the context is defined on that line"
    (fn [assert-that]
      (let [criteria (query/defined-on-line 0)
            a (create-context {::definition-line-number 0
                               ::components [(create-test {::definition-line-number 1})
                                             (create-test {::definition-line-number 2})]})
            b (create-context {::definition-line-number 3
                               ::components [(create-test {::definition-line-number 4})
                                             (create-test {::definition-line-number 5})]})
            defined-components [a b]
            result (query/query criteria defined-components)]
        (assert-that (returned-components result)
                     (c/is [a]))))))

(c/describe `query/description-matches
  (c/it "returns top-level tests whose description matches the given regex"
    (fn [assert-that]
      (let [criteria (query/description-matches #"foo")
            a (create-test {::description "foo"})
            b (create-test {::description "bar"})
            defined-components [a b]
            result (query/query criteria defined-components)]
        (assert-that (returned-components result)
                     (c/is [a])))))

  (c/it "does not return empty contexts whose subject matches the regex"
    (fn [assert-that]
      (let [criteria (query/description-matches #"foo")
            a (create-context {::subject "foo"})
            b (create-context {::subject "bar"})
            defined-components [a b]
            result (query/query criteria defined-components)]
        (assert-that (returned-components result)
                     (c/is [])))))

  (c/it "returns all tests in a context whose subject matches the regex"
    (fn [assert-that]
      (let [criteria (query/description-matches #"foo.*")
            a (create-context {::subject "foo"
                               ::components [(create-test)
                                             (create-test)]})
            b (create-context {::subject "bar"
                               ::components [(create-test)
                                             (create-test)]})
            defined-components [a b]
            result (query/query criteria defined-components)]
        (assert-that (returned-components result)
                     (c/is [a])))))

  (c/it "returns tests whose description combined with their contexts subject matches the regex"
    (fn [assert-that]
      (let [criteria (query/description-matches #"a a")
            t1 (create-test {::description "a"})
            t2 (create-test {::description "b"})
            t3 (create-test {::description "a"})
            t4 (create-test {::description "b"})
            a (create-context {::subject "a"
                               ::components [t1 t2]})
            b (create-context {::subject "b"
                               ::components [t3 t4]})
            defined-components [a b]
            result (query/query criteria defined-components)]
        (assert-that (returned-components result)
                     (c/is [(assoc-in a [::c/sub-components ::c/components] [t1])])))))

  (c/it "returns tests whose full description contains non-string values if the str value matches"
    (fn [assert-that]
      (let [criteria (query/description-matches #".*#'coconut\.v1\.query/query.*")
            t1 (create-test {::description "a"})
            t2 (create-test {::description "b"})
            t3 (create-test {::description "a"})
            t4 (create-test {::description "b"})
            a (create-context {::subject #'coconut.v1.query/query
                               ::components [t1 t2]})
            b (create-context {::subject "b"
                               ::components [t3 t4]})
            defined-components [a b]
            result (query/query criteria defined-components)]
        (assert-that (returned-components result)
                     (c/is [a]))))))
