(ns coconut.v1.html
  (:require-macros [cljs.core.async.macros :as async])
  (:require
    [clojure.core.async :as async :refer [<!]]
    [coconut.v1.rendering :as rendering]
    [clojure.string :as cs]
    [coconut.v1.platform :as platform]
    [coconut.v1.query :as query]
    [coconut.v1.running :as running]
    [coconut.v1.summarizing :as summarizing]
    [coconut.v1.documentation :as documentation]
    [coconut.v1.util :as util]
    ))

(defn ^{:private true} color-string
  ([segment]
   (case (::rendering/color segment)
     ::rendering/grey "grey"
     ::rendering/red "red"
     ::rendering/green "green"
     ::rendering/yellow "yellow"
     ::rendering/teal "teal"
     nil ""
     (throw (platform/illegal-argument-exception
              (str "unsupported color: "
                   (::rendering/color segment)))))))

(defn ^{:private true} scroll-to-bottom
  ([]
   (.scrollTo js/window
              0
              js/document.body.scrollHeight)))

(defn ^{:private true} create-element
  ([tag]
   (js/document.createElement (name tag))))

(defn ^{:private true} set-inner-html
  ([element html]
   (aset element "innerHTML" html)))

(defn ^{:private true} set-text
  ([element text]
   (set-inner-html element
                   (cs/replace text #" " "&nbsp"))))

(defn ^{:private true} set-class
  ([element class-name]
   (aset element "className" (name class-name))))

(defn ^{:private true} add-to
  ([child parent]
   (.appendChild parent child)))

(defmulti ^{:private true} print-output
  (fn [container data]
    (::rendering/type data)))

(defmethod ^{:private true} print-output
  ::rendering/newline
  ([container data]
   (doto (create-element :br)
     (add-to container))))

(defmethod ^{:private true} print-output
  ::rendering/line
  ([container data]
   (let [paragraph (create-element :p)]
     (doseq [segment (::rendering/segments data)]
       (doto (create-element :span)
         (set-text (::rendering/text segment))
         (set-class (color-string segment))
         (add-to paragraph)))
     (doto paragraph
       (add-to container))
     (scroll-to-bottom))))

(defmethod ^{:private true} print-output
  ::rendering/segment
  ([container data]
   (let [paragraph (create-element :span)]
     (doseq [segment (::rendering/segments data)]
       (doto (create-element :span)
         (set-text (::rendering/text segment))
         (set-class (color-string segment))
         (add-to paragraph)))
     (doto paragraph
       (add-to container))
     (scroll-to-bottom))))

(defn output
  "Given a core.async channel of output, renders the html and
  appends it to the document."
  ([channel]
   (let [container (doto (create-element :div)
                     (set-class :container)
                     (add-to js/document.body))]
     ((util/channel-consuming (partial print-output container)) channel))))
