(ns coconut.v1.summarizing
  (:require
    [coconut.v1.running :as running]
    [coconut.v1.aggregation :as aggregation]
    ))

(defmulti summarize-event
  (fn [state event]
    [(::current-status (::current-test state))
     (::running/type event)]))

(defmethod summarize-event
  [nil ::running/suite-started]
  ([state event]
   #::aggregation{:events [#::{:type ::suite-started
                               :total-number-of-tests (::running/total-number-of-tests event)}]
                  :state #::{:context []
                             :current-test nil
                             :suite-started-at (::running/current-time-millis event)}}))

(defmethod summarize-event
  [nil ::running/context-started]
  ([state event]
   #::aggregation{:events [#::{:type ::context-started
                               :subject (::running/subject event)}]
                  :state (update state
                                 ::context
                                 conj
                                 (::running/subject event))}))

(defmethod summarize-event
  [nil ::running/test-marked-pending]
  ([state event]
   #::aggregation{:events [#::{:type ::test-pending
                               :definition-line-number (::running/definition-line-number event)
                               :context (::context state)
                               :description (::running/description event)
                               :pending-reason (::running/pending-reason event)
                               :duration-in-milliseconds 0}]
                  :state state}))

(defmethod summarize-event
  [nil ::running/test-started]
  ([state event]
   #::aggregation{:events []
                  :state (assoc state
                                ::current-test #::{:current-status ::nothing-asserted
                                                   :relevant-assertion-failed-event nil
                                                   :namespace-name (::running/namespace-name event)
                                                   :definition-line-number (::running/definition-line-number event)})}))

(defmethod summarize-event
  [::nothing-asserted ::running/assertion-passed]
  ([state event]
   #::aggregation{:events []
                  :state (assoc-in state
                                   [::current-test ::current-status] ::passing)}))

(defmethod summarize-event
  [::nothing-asserted ::running/assertion-failed]
  ([state event]
   #::aggregation{:events []
                  :state (-> state
                             (assoc-in [::current-test ::current-status] ::failing)
                             (assoc-in [::current-test ::relevant-assertion-failed-event] event))}))

(defmethod summarize-event
  [::nothing-asserted ::running/test-timed-out]
  ([state event]
   #::aggregation{:events [#::{:type ::test-timed-out
                               :context (::context state)
                               :namespace-name (::namespace-name (::current-test state))
                               :definition-line-number (::definition-line-number (::current-test state))
                               :description (::running/description event)}]
                  :state (assoc state
                                ::current-test nil)}))

(defmethod summarize-event
  [::nothing-asserted ::running/test-threw-exception]
  ([state event]
   #::aggregation{:events [#::{:type ::test-threw-exception
                               :namespace-name (::namespace-name (::current-test state))
                               :definition-line-number (::definition-line-number (::current-test state))
                               :context (::context state)
                               :description (::running/description event)
                               :exception (::running/exception event)}]
                  :state (assoc state
                                ::current-test nil)}))

(defmethod summarize-event
  [::nothing-asserted ::running/test-finished]
  ([state event]
   #::aggregation{:events [#::{:type ::test-pending
                               :definition-line-number (::definition-line-number (::current-test state))
                               :context (::context state)
                               :description (::running/description event)
                               :pending-reason "nothing has been asserted"}]
                  :state (assoc state
                                ::current-test nil)}))

(defmethod summarize-event
  [::failing ::running/assertion-passed]
  ([state event]
   #::aggregation{:events []
                  :state state}))

(defmethod summarize-event
  [::failing ::running/assertion-failed]
  ([state event]
   #::aggregation{:events []
                  :state state}))

(defmethod summarize-event
  [::failing ::running/test-timed-out]
  ([state event]
   #::aggregation{:events [#::{:type ::test-timed-out
                               :context (::context state)
                               :namespace-name (::namespace-name (::current-test state))
                               :definition-line-number (::definition-line-number (::current-test state))
                               :description (::running/description event)}]
                  :state (assoc state
                                ::current-test nil)}))

(defmethod summarize-event
  [::failing ::running/test-threw-exception]
  ([state event]
   #::aggregation{:events [#::{:type ::test-failed
                               :namespace-name (::namespace-name (::current-test state))
                               :definition-line-number (::definition-line-number (::current-test state))
                               :context (::context state)
                               :description (::running/description event)
                               :expected (::running/expected (::relevant-assertion-failed-event (::current-test state)))
                               :actual (::running/actual (::relevant-assertion-failed-event (::current-test state)))}]
                  :state (assoc state
                                ::current-test nil)}))

(defmethod summarize-event
  [::failing ::running/test-finished]
  ([state event]
   #::aggregation{:events [#::{:type ::test-failed
                               :namespace-name (::namespace-name (::current-test state))
                               :definition-line-number (::definition-line-number (::current-test state))
                               :context (::context state)
                               :description (::running/description event)
                               :expected (::running/expected (::relevant-assertion-failed-event (::current-test state)))
                               :actual (::running/actual (::relevant-assertion-failed-event (::current-test state)))}]
                  :state (assoc state
                                ::current-test nil)}))

(defmethod summarize-event
  [::passing ::running/assertion-passed]
  ([state event]
   #::aggregation{:events []
                  :state state}))

(defmethod summarize-event
  [::passing ::running/assertion-failed]
  ([state event]
   #::aggregation{:events []
                  :state (-> state
                             (assoc-in [::current-test ::current-status] ::failing)
                             (assoc-in [::current-test ::relevant-assertion-failed-event] event))}))

(defmethod summarize-event
  [::passing ::running/test-timed-out]
  ([state event]
   #::aggregation{:events [#::{:type ::test-timed-out
                               :context (::context state)
                               :namespace-name (::namespace-name (::current-test state))
                               :definition-line-number (::definition-line-number (::current-test state))
                               :description (::running/description event)}]
                  :state (assoc state
                                ::current-test nil)}))

(defmethod summarize-event
  [::passing ::running/test-threw-exception]
  ([state event]
   #::aggregation{:events [#::{:type ::test-threw-exception
                               :namespace-name (::namespace-name (::current-test state))
                               :definition-line-number (::definition-line-number (::current-test state))
                               :context (::context state)
                               :description (::running/description event)
                               :exception (::running/exception event)}]
                  :state (assoc state
                                ::current-test nil)}))

(defmethod summarize-event
  [::passing ::running/test-finished]
  ([state event]
   #::aggregation{:events [#::{:type ::test-passed
                               :namespace-name (::namespace-name (::current-test state))
                               :definition-line-number (::definition-line-number (::current-test state))
                               :context (::context state)
                               :description (::running/description event)}]
                  :state (assoc state
                                ::current-test nil)}))

(defmethod summarize-event
  [nil ::running/context-finished]
  ([state event]
   #::aggregation{:events [#::{:type ::context-finished
                               :subject (::running/subject event)}]
                  :state (update state ::context pop)}))

(defmethod summarize-event
  [nil ::running/suite-finished]
  ([state event]
   #::aggregation{:events [#::{:type ::suite-finished
                               :duration-in-milliseconds (- (::running/current-time-millis event)
                                                            (::suite-started-at state))}]
                  :state state}))

(def summarize
  "Given a core.async channel of test events, returns a new channel of
  events which indicates the outcome of the suite and its contained contexts
  and tests."
  (aggregation/aggregation summarize-event))
