(ns coconut.v1.running-test
  #?(:cljs (:require-macros [cljs.core.async.macros :as async]))
  (:require
    [clojure.core.async :as async :refer [<!]]
    [coconut.v1.core :as c]
    [coconut.v1.query :as query]
    [coconut.v1.running :as running]
    [coconut.v1.test-helpers :as helpers]
    [coconut.v1.test-namespaces.fixtures]
    ))

(defn run-channel
  ([component]
   (->> component
        (query/return)
        (running/run))))

(defn run
  ([component]
   (async/into []
               (run-channel component))))

(def number-of-events
  (helpers/number-of-events-function ::running/type))

(defn run-fixtures
  ([]
   (->> (helpers/load-fixtures)
        (query/return-all)
        (running/run)
        (async/into []))))

(c/describe `running/run
  {:asynchronous {:timeout 250}}

  (c/for [component (helpers/load-fixtures)]
    (c/it (str "emits events for " (c/string-label component))
      (fn [assert-that done]
        (async/go
          (let [event-channel (run-channel component)]
            (doseq [data (::running/event-data component)]
              (assert-that (<! event-channel)
                           (c/contains data)))
            (assert-that (<! event-channel)
                         (c/is nil))
          (done))))))

  (c/let [fixture-events (helpers/go->promise-channel (run-fixtures))]
    (c/it "includes an incrementing timestamp for every event"
      (fn [assert-that done]
        (async/go
          (let [events (<! fixture-events)
                timestamps (into (vector)
                                 (map ::running/current-time-millis)
                                 events)]
            ;; every event has a timestamp
            (assert-that (count timestamps)
                         (c/is (count events)))
            ;; every timestamp is a number
            (assert-that timestamps
                         (c/satisfies (partial every? number?)))
            ;; the timestamps are not some static value
            (assert-that (count (set timestamps))
                         (c/is-not 1))
            ;; the timestamps are incrementing
            (assert-that timestamps
                         (c/is (sort timestamps)))
            (done)))))

    (c/it "includes the total number of tests in the suite started event"
      (fn [assert-that done]
        (async/go
          (let [events (<! fixture-events)]
            (assert-that (::running/total-number-of-tests (first events))
                         (c/is (+ (number-of-events events ::running/test-started)
                                  (number-of-events events ::running/test-marked-pending))))
            (done)))))

    (c/it "includes the subject in the context started event"
      (fn [assert-that done]
        (async/go
          (let [component (c/context "foo"
                            (c/it "..." (fn [_])))
                events (<! (run component))]
            (doseq [event events]
              (when (= ::running/context-started (::running/type event))
                (assert-that (::running/subject event)
                             (c/is "foo"))))
            (done)))))

    (c/it "includes the subject in the context finished event"
      (fn [assert-that done]
        (async/go
          (let [component (c/context "foo"
                            (c/it "..." (fn [_])))
                events (<! (run component))]
            (doseq [event events]
              (when (= ::running/context-finished (::running/type event))
                (assert-that (::running/subject event)
                             (c/is "foo"))))
            (done)))))

    (c/it "includes the test description in the test marked pending event"
      (fn [assert-that done]
        (async/go
          (let [component (c/context "foo"
                            (c/it "bar"
                              {:pending true}
                              (fn [_])))
                events (<! (run component))]
            (doseq [event events]
              (when (= ::running/test-marked-pending (::running/type event))
                (assert-that (::running/description event)
                             (c/is "bar"))))
            (done)))))

    (c/it "includes the pending reason in the test marked pending event"
      (fn [assert-that done]
        (async/go
          (let [component (c/context "foo"
                            (c/it "bar"
                              {:pending "for some reason"}
                              (fn [_])))
                events (<! (run component))]
            (doseq [event events]
              (when (= ::running/test-marked-pending (::running/type event))
                (assert-that (::running/pending-reason event)
                             (c/is "for some reason"))))
            (done)))))))
