(ns aclaimant.server.core.content-type
  (:require
    [aclaimant.server.core.request :as request]
    [camel-snake-kebab.core :as csk]
    [camel-snake-kebab.extras :as csk-extras]
    [cheshire.core :as json]))

(defn- ->json-keys [body]
  (if (string? body)
    body
    (-> body
        request/jsonify-keys
        (json/generate-string {:escape-non-ascii true}))))

(defn ->kebab-case-keyword [thing]
  (if-let [thing-ns (and (keyword? thing) (namespace thing))]
    (keyword thing-ns (csk/->kebab-case-string thing))
    (csk/->kebab-case-keyword thing)))

(defn ^:private ->edn [body]
  (pr-str
    (if (string? body)
      (csk-extras/transform-keys ->kebab-case-keyword (json/parse-string body))
      body)))

(defn transform-and-content-type [accept]
  (condp re-find (str accept)
    #"^application/edn" :edn
    #"^application/json" :json
    nil))

(def content-types
  {:json [->json-keys "application/json"]
   :edn [->edn "application/edn"] })

(defn wrap-content-type-response
  "Responds with a client-specified content-type"
  [handler & [default-accept]]
  (fn [request]
    (let [{:keys [headers] :as response} (handler request)]
      (if (contains? #{"text/html" "text/plain" "application/pdf"} (get headers "Content-Type"))
        response
        (let [default-accept (or default-accept :json)
              accept (get-in request [:headers "accept"])
              [transformer content-type] (content-types
                                           (or (transform-and-content-type accept) default-accept))]
          (-> response
              (update :body transformer)
              (update :headers assoc "Content-Type" content-type)))))))
