(ns com.adgoji.api.displayvideo.client
  (:require
   [clojure.java.io :as io]
   [com.adgoji.api.displayvideo.advertisers :as advertisers]
   [com.adgoji.api.utils.base64 :as base64])
  (:import
   (com.google.api.client.auth.oauth2 Credential)
   (com.google.api.client.extensions.java6.auth.oauth2 AuthorizationCodeInstalledApp)
   (com.google.api.client.extensions.jetty.auth.oauth2 LocalServerReceiver)
   (com.google.api.client.googleapis.auth.oauth2 GoogleAuthorizationCodeFlow$Builder GoogleClientSecrets GoogleCredential)
   (com.google.api.client.googleapis.util Utils)
   (com.google.api.client.http HttpRequest HttpRequestInitializer)
   (com.google.api.services.displayvideo.v2 DisplayVideo DisplayVideo$Builder DisplayVideoScopes)))

(set! *warn-on-reflection* true)

(def ->scope
  {:display-video                 DisplayVideoScopes/DISPLAY_VIDEO
   :doubleclickbidmanager         DisplayVideoScopes/DOUBLECLICKBIDMANAGER
   :display-video-mediaplanning   DisplayVideoScopes/DISPLAY_VIDEO_MEDIAPLANNING
   :display-video-user-management DisplayVideoScopes/DISPLAY_VIDEO_USER_MANAGEMENT})

(defn- convert-scopes
  [scopes]
  (if (coll? scopes)
    (let [scopes-xf
          (cond->> (filter (partial contains? (set (DisplayVideoScopes/all))))
            (every? keyword? scopes) (comp (map ->scope)))]
      (into #{} scopes-xf scopes))
    (if (= :all scopes)
      (set (DisplayVideoScopes/all))
      (throw (ex-info "Invalid scopes" {:scopes scopes})))))

(defmulti google-credential :mode :default ::default)

(defmethod google-credential :service-base64
  ^Credential
  [{:keys [credential scopes]}]
  (with-open [credential-bytes (-> credential
                                   (base64/decode)
                                   (.getBytes)
                                   (io/input-stream))]
    (-> credential-bytes
        (GoogleCredential/fromStream)
        (.createScoped (convert-scopes scopes)))))

(defmethod google-credential :service-file
  ^Credential
  [{:keys [credential scopes]}]
  (with-open [credential-file (io/input-stream (io/file credential))]
    (-> credential-file
        (GoogleCredential/fromStream)
        (.createScoped (convert-scopes scopes)))))

(defmethod google-credential :installed
  [{:keys [credential scopes]}]
  ^Credential
  (with-open [credentials-file (io/reader (io/file credential))]
    (let [secrets (GoogleClientSecrets/load (Utils/getDefaultJsonFactory) credentials-file)
          flow    (-> (GoogleAuthorizationCodeFlow$Builder.
                       (Utils/getDefaultTransport)
                       (Utils/getDefaultJsonFactory)
                       secrets
                       (convert-scopes scopes))
                      (.build))]
      (-> (AuthorizationCodeInstalledApp. flow (LocalServerReceiver.))
          (.authorize "user")))))

(defmethod google-credential ::default
  [{:keys [mode]}]
  (throw (ex-info "Unknown mode" {:mode mode})))

(defn- set-http-timeout
  [^GoogleCredential google-credential ^long connect-timeout-ms ^long read-timeout-ms]
  (reify
    HttpRequestInitializer
    (^void initialize [_this ^HttpRequest http-request]
      (.initialize google-credential http-request)
      (.setConnectTimeout http-request connect-timeout-ms)
      (.setReadTimeout http-request read-timeout-ms))))

(defn new-client
  ^DisplayVideo
  [^GoogleCredential google-credential
   ^String application-name
   {:keys [connect-timeout-ms read-timeout-ms]
    :or   {connect-timeout-ms 60000
           read-timeout-ms    60000}}]
  (-> (DisplayVideo$Builder. (. google-credential getTransport)
                             (. google-credential getJsonFactory)
                             (set-http-timeout google-credential
                                               connect-timeout-ms
                                               read-timeout-ms))
      (.setApplicationName application-name)
      (.build)))

(comment
  ;; Read service account credentials from file
  (let [creds  (google-credential {:mode       :service-file
                                   :credential "/Users/rrudakov/Work/adgoji-dv360-api-108e7f7c4a66.json"
                                   :scopes     :all})
        client (new-client creds "adgoji-dv360-api-client" {})]
    (advertisers/get-list client
                          1354604054
                          {:order-by    "entityStatus"
                           :filter-expr "entityStatus=\"ENTITY_STATUS_ACTIVE\""}))

  ;; Read service account credentials from base64 string
  (let [creds (google-credential {:mode       :service-base64
                                  :credential "encoded-credential"
                                  :scopes     [:display-video]})]
    (new-client creds "adgoji-dv360-api-client" {}))

  ;; Use oauth2 browser flow from file (aka installed app flow)
  (let [creds  (google-credential {:mode       :installed
                                   :credential "/Users/rrudakov/Work/oauth_dv360_client_secret_981639147489-fpbmoafklq7lljjsk3ktc5factge0rr4.apps.googleusercontent.com.json"
                                   :scopes     #{:display-video
                                                 :doubleclickbidmanager
                                                 :display-video-mediaplanning}})
        client (new-client creds "adgoji-dv360-api-client" {})]
    (advertisers/get-list client 1354604054 {})))
