(ns com.adgoji.api.utils.java
  (:require
   [camel-snake-kebab.core :as csk]
   [clojure.java.data :as j])
  (:import
   (java.lang.reflect Method)))

(set! *warn-on-reflection* true)

(defmulti clj->java
  "Adapted version of [[clojure.java.data/to-java]].

  Google models are not proper java beans and `to-java` function
  cannot find setter methods successfully."
  (fn [destination-type value] [destination-type (class value)]))

(defn- find-setter-method
  [^Class cls method-name]
  (when-let [methods
             (->> (.getMethods cls)
                  (sequence (filter (fn [^Method method]
                                      (= (.getName method) method-name))))
                  (seq))]
    (if (not= 1 (count methods))
      (throw (ex-info (format "More than one %s method found" method-name) {}))
      (let [method           ^Method (first methods)
            parameters-count (count (.getParameterTypes method))]
        (if (not= 1 parameters-count)
          (throw (ex-info "Setter %s should take 1 argument, but it takes %d"
                          method-name
                          parameters-count))
          method)))))

(defn- get-setter-type
  [^java.lang.reflect.Method method]
  (get (.getParameterTypes method) 0))

(defn- set-prop
  [^Object instance [k v]]
  (let [clazz       (.getClass instance)
        setter-name (str "set" (csk/->PascalCaseString k))]
    (if-let [setter-method ^Method (find-setter-method clazz setter-name)]
      (.invoke setter-method
               instance
               (into-array [(clj->java (get-setter-type setter-method) v)]))
      instance)))

(defmethod clj->java :default [^Class cls value]
  (j/to-java cls value))

(defmethod clj->java [java.lang.Object clojure.lang.APersistentMap]
  [^Class clazz props]
  (let [instance (.newInstance clazz)]
    (reduce set-prop instance props)))
