(ns com.adgoji.line-items.core
  (:require
   [com.adgoji.displayvideo-spec.interface :as displayvideo-spec]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.sdk-client-utils.interface :as sdk-client-utils]
   [com.adgoji.spec.interface :as spec])
  (:import
   (com.google.api.services.displayvideo.v2 DisplayVideo DisplayVideo$Advertisers$LineItems$BulkListAssignedTargetingOptions DisplayVideo$Advertisers$LineItems$List)
   (com.google.api.services.displayvideo.v2.model BulkEditAssignedTargetingOptionsRequest BulkUpdateLineItemsRequest LineItem)))

(defn get-by-id
  [^DisplayVideo client advertiser-id line-item-id]
  (-> client
      (.advertisers)
      (.lineItems)
      (.get (long advertiser-id) (long line-item-id))
      (sdk-client-utils/execute-and-convert)
      (spec/check ::displayvideo-spec/line-item)))

(defn create
  [^DisplayVideo client advertiser-id line-item]
  (let [line-item-conformed (spec/check line-item ::displayvideo-spec/line-item)
        line-item-request   (java-utils/clj->java LineItem line-item-conformed)]
    (-> client
        (.advertisers)
        (.lineItems)
        (.create (long advertiser-id) line-item-request)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::displayvideo-spec/line-item))))

(defn delete
  [^DisplayVideo client advertiser-id line-item-id]
  (-> client
      (.advertisers)
      (.lineItems)
      (.delete (long advertiser-id) (long line-item-id))
      (sdk-client-utils/execute-and-convert)))

(defn update-by-id
  [^DisplayVideo client advertiser-id line-item-id line-item update-mask]
  (let [mask              (-> update-mask
                              (spec/check ::displayvideo-spec/update-mask)
                              sdk-client-utils/normalize-mask)
        line-item-request (java-utils/clj->java LineItem line-item)]
    (-> client
        (.advertisers)
        (.lineItems)
        (.patch (long advertiser-id)
                (long line-item-id)
                line-item-request)
        (.setUpdateMask mask)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::displayvideo-spec/line-item))))

(defn- fetch-line-items-page
  [^DisplayVideo$Advertisers$LineItems$List request ^String next-page-token]
  (-> request
      (.setPageToken next-page-token)
      (sdk-client-utils/execute-and-convert)))

(defn get-list
  [^DisplayVideo client advertiser-id {:keys [order-by filter-expr limit]}]
  (let [request (-> client
                    (.advertisers)
                    (.lineItems)
                    (.list (long advertiser-id)))]
    (when order-by
      (.setOrderBy ^DisplayVideo$Advertisers$LineItems$List request order-by))
    (when filter-expr
      (.setFilter ^DisplayVideo$Advertisers$LineItems$List request filter-expr))
    (-> (sdk-client-utils/execute-and-convert-paginated
         (partial fetch-line-items-page request)
         :line-items
         limit)
        (spec/check ::displayvideo-spec/line-items))))

(defn bulk-edit-assigned-targeting-options
  [^DisplayVideo client advertiser-id bulk-edit-assigned-targeting-options-request]
  (let [request-conformed
        (spec/check bulk-edit-assigned-targeting-options-request
                    ::displayvideo-spec/bulk-edit-assigned-options-request)

        request-body
        (java-utils/clj->java BulkEditAssignedTargetingOptionsRequest
                              request-conformed)]
    (-> client
        (.advertisers)
        (.lineItems)
        (.bulkEditAssignedTargetingOptions (long advertiser-id) request-body)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::displayvideo-spec/bulk-edit-assigned-options-response))))

(defn fetch-assigned-targeting-options-page
  [^DisplayVideo$Advertisers$LineItems$BulkListAssignedTargetingOptions request
   ^String next-page-token]
  (-> request
      (.setPageToken next-page-token)
      (sdk-client-utils/execute-and-convert)))

(defn bulk-list-assigned-targeting-options
  [^DisplayVideo client advertiser-id {:keys [line-item-ids
                                              order-by
                                              filter-expr
                                              limit]}]
  (let [request (-> client
                    (.advertisers)
                    (.lineItems)
                    (.bulkListAssignedTargetingOptions (long advertiser-id))
                    (.setLineItemIds (into [] (map long) line-item-ids)))]

    (when order-by
      (.setOrderBy
       ^DisplayVideo$Advertisers$LineItems$BulkListAssignedTargetingOptions
       request
       order-by))
    (when filter-expr
      (.setFilter
       ^DisplayVideo$Advertisers$LineItems$BulkListAssignedTargetingOptions
       request
       filter-expr))
    (-> (sdk-client-utils/execute-and-convert-paginated
         (partial fetch-assigned-targeting-options-page request)
         :line-item-assigned-targeting-options
         limit)
        (spec/check ::displayvideo-spec/bulk-list-assigned-options-response))))

(defn bulk-update
  [^DisplayVideo client advertiser-id bulk-update-request-body]
  (let [request-conformed (-> bulk-update-request-body
                              (spec/check ::displayvideo-spec/bulk-update-line-items-request)
                              (update :update-mask sdk-client-utils/normalize-mask))
        request           (java-utils/clj->java BulkUpdateLineItemsRequest
                                                request-conformed)]
    (-> client
        (.advertisers)
        (.lineItems)
        (.bulkUpdate (long advertiser-id) request)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::displayvideo-spec/bulk-update-line-items-response))))
