(ns com.adgoji.advertisers.core
  (:require
   [camel-snake-kebab.core :as csk]
   [clojure.string :as str]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.sdk-client-utils.interface :as sdk-client-utils]
   [com.adgoji.spec.interface :as spec])
  (:import
   (com.google.api.services.displayvideo.v2 DisplayVideo DisplayVideo$Advertisers$List)
   (com.google.api.services.displayvideo.v2.model Advertiser)))

(defn- fetch-advertisers-page
  [^DisplayVideo$Advertisers$List request ^String next-page-token]
  (-> request
      (.setPageToken next-page-token)
      (sdk-client-utils/execute-and-convert)))

(defn get-list
  "Returns a list of available advertisers by `partner-id`."
  [^DisplayVideo client partner-id {:keys [order-by filter-expr limit]}]
  (let [request (-> client
                    (.advertisers)
                    (.list)
                    (.setPartnerId partner-id))]
    (when order-by
      (.setOrderBy ^DisplayVideo$Advertisers$List request order-by))
    (when filter-expr
      (.setFilter ^DisplayVideo$Advertisers$List request filter-expr))
    (-> (sdk-client-utils/execute-and-convert-paginated
         (partial fetch-advertisers-page request)
         :advertisers
         limit)
        (spec/check ::spec/advertisers))))

(defn get-by-id
  [^DisplayVideo client advertiser-id]
  (-> client
      (.advertisers)
      (.get advertiser-id)
      (sdk-client-utils/execute-and-convert)
      (spec/check ::spec/advertiser)))

(defn create
  [^DisplayVideo client advertiser]
  (let [advertiser-conformed (spec/check advertiser ::spec/advertiser)
        advertiser-request   (java-utils/clj->java Advertiser
                                                   advertiser-conformed)]
    (-> client
        (.advertisers)
        (.create advertiser-request)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::spec/advertiser))))

(defn delete
  [^DisplayVideo client advertiser-id]
  (-> client
      (.advertisers)
      (.delete advertiser-id)
      (sdk-client-utils/execute-and-convert)))

(defn update-by-id
  [^DisplayVideo client advertiser-id advertiser update-mask]
  (let [mask-conformed     (spec/check update-mask ::spec/update-mask)
        mask               (->> mask-conformed
                                (sequence (map csk/->camelCaseString))
                                (str/join \,))
        advertiser-request (java-utils/clj->java Advertiser advertiser)]
    (-> client
        (.advertisers)
        (.patch advertiser-id advertiser-request)
        (.setUpdateMask mask)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::spec/advertiser))))
