(ns com.adgoji.java-utils.core
  (:require
   [camel-snake-kebab.core :as csk]
   [clojure.java.data :as j])
  (:import
   (java.lang.reflect Method ParameterizedType)))

(defmulti clj->java
  "Adapted version of [[clojure.java.data/to-java]].

  Google models are not proper java beans and `to-java` function
  cannot find setter methods successfully."
  (fn [destination-type value] [destination-type (class value)]))

(defn- find-setter-method
  [^Class cls ^String method-name]
  (when-let [methods
             (->> (.getMethods cls)
                  (sequence (filter (fn [^Method method]
                                      (= (.getName method) method-name))))
                  (seq))]
    (if (not= 1 (count methods))
      (throw (ex-info (format "More than one %s method found" method-name) {}))
      (let [method           ^Method (first methods)
            parameters-count (count (.getParameterTypes method))]
        (if (not= 1 parameters-count)
          (throw (ex-info (format "Setter %s should take 1 argument, but it takes %d"
                                  method-name
                                  parameters-count)
                          {}))
          method)))))

(defn- get-setter-type
  ^Class
  [^Method method]
  (get (.getParameterTypes method) 0))

(defn- get-list-setter-type
  ^Class
  [^Method method]
  (let [list-type (-> method
                      (.getGenericParameterTypes)
                      (get 0))]
    (if (instance? ParameterizedType list-type)
      (-> ^ParameterizedType list-type
          (.getActualTypeArguments)
          (get 0))
      Object)))

(defn- set-prop
  [^Object instance [k v]]
  (let [clazz       (.getClass instance)
        setter-name (str "set" (csk/->PascalCaseString k))]
    (if-let [setter-method ^Method (find-setter-method clazz setter-name)]
      (let [setter-type (get-setter-type setter-method)]
        (if (= java.util.List setter-type)
          (let [list-type        (get-list-setter-type setter-method)
                converted-values (into []
                                       (map (fn [item] (clj->java list-type item)))
                                       v)]
            (.invoke setter-method
                     instance
                     (into-array [converted-values])))
          (.invoke setter-method
                   instance
                   (into-array [(clj->java (get-setter-type setter-method) v)]))))
      instance)))

(defmethod clj->java :default [^Class cls value]
  (j/to-java cls value))

(defmethod clj->java [java.lang.Object clojure.lang.APersistentMap]
  [^Class clazz props]
  (let [instance (.newInstance clazz)]
    (reduce set-prop instance props)))
