(ns com.adgoji.line-items.core
  (:require
   [camel-snake-kebab.core :as csk]
   [clojure.string :as str]
   [com.adgoji.java-utils.interface :as java-utils]
   [com.adgoji.sdk-client-utils.interface :as sdk-client-utils]
   [com.adgoji.spec.interface :as spec])
  (:import
   (com.google.api.services.displayvideo.v2 DisplayVideo DisplayVideo$Advertisers$LineItems$List)
   (com.google.api.services.displayvideo.v2.model LineItem)))

(defn get-by-id
  [^DisplayVideo client advertiser-id lineitem-id]
  (-> client
      (.advertisers)
      (.lineItems)
      (.get advertiser-id lineitem-id)
      (sdk-client-utils/execute-and-convert)
      (spec/check ::spec/line-item)))

(defn create
  [^DisplayVideo client advertiser-id lineItem]
  (let [line-item-conformed (spec/check lineItem ::spec/line-item)
        line-item-request   (java-utils/clj->java LineItem line-item-conformed)]
    (-> client
        (.advertisers)
        (.lineItems)
        (.create advertiser-id line-item-request)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::spec/line-item))))

(defn delete
  [^DisplayVideo client advertiser-id line-item-id]
  (-> client
      (.advertisers)
      (.lineItems)
      (.delete advertiser-id line-item-id)
      (sdk-client-utils/execute-and-convert)))

(defn update-by-id
  [^DisplayVideo client advertiser-id line-item-id line-item update-mask]
  (let [mask-conformed    (spec/check update-mask ::spec/update-mask)
        mask              (->> mask-conformed
                               (sequence (map csk/->camelCaseString))
                               (str/join \,))
        line-item-request (java-utils/clj->java LineItem line-item)]
    (-> client
        (.advertisers)
        (.lineItems)
        (.patch advertiser-id line-item-id line-item-request)
        (.setUpdateMask mask)
        (sdk-client-utils/execute-and-convert)
        (spec/check ::spec/line-item))))

(defn- fetch-line-items-page
  [^DisplayVideo$Advertisers$LineItems$List request ^String next-page-token]
  (-> request
      (.setPageToken next-page-token)
      (sdk-client-utils/execute-and-convert)))

(defn get-list
  [^DisplayVideo client advertiser-id {:keys [order-by filter-expr limit]}]
  (let [request (-> client
                    (.advertisers)
                    (.lineItems)
                    (.list advertiser-id))]
    (when order-by
      (.setOrderBy ^DisplayVideo$Advertisers$LineItems$List request order-by))
    (when filter-expr
      (.setFilter ^DisplayVideo$Advertisers$LineItems$List request filter-expr))
    (-> (sdk-client-utils/execute-and-convert-paginated (partial fetch-line-items-page request)
                                                        :line-items
                                                        limit)
        (spec/check ::spec/line-items))))
