(ns com.adgoji.maps-utils.core
  (:import
   (com.google.maps.model
    AddressComponent
    AddressComponentType
    AddressType
    Bounds
    GeocodingResult
    Geometry
    LatLng
    PlusCode)))

(defn clj-lat-lng->java-lat-lng [clj-lat-lng]
  (new LatLng (:lat clj-lat-lng) (:lng clj-lat-lng)))

(defn- parse-address-component-types [types]
  (into []
        (map (fn [^AddressComponentType t] (.toString t)))
        types))

(defn- parse-address-types [types]
  (into []
        (map (fn [^AddressType t] (.toString t)))
        types))

(defn- parse-address-component [^AddressComponent  address-component]
  {:long-name  (.longName address-component)
   :short-name (.shortName address-component)
   :types      (parse-address-component-types (.types address-component))})

(defn- parse-lat-lng [^LatLng lat-lng]
  {:lat (.lat lat-lng)
   :lng (.lng lat-lng)})

(defn- parse-bounds [^Bounds bounds]
  {:northeast (parse-lat-lng (.northeast bounds))
   :southwest (parse-lat-lng (.southwest bounds))})

(defn- parse-geometry [^Geometry geometry]
  (let [bounds        (.bounds geometry)
        location      (.location geometry)
        location-type (.locationType geometry)
        viewport      (.viewport geometry)]
    (cond-> {:location-type (.toUrlValue location-type)}
      bounds   (assoc :bounds (parse-bounds bounds))
      location (assoc :location (parse-lat-lng location))
      viewport (assoc :viewport (parse-bounds viewport)))))

(defn- parse-plus-code [^PlusCode code]
  {:compound-code  (.compoundCode code)
   :global-code    (.globalCode code)})

(defn geocoding-result->map [^GeocodingResult result]
  (let [plus-code           (.plusCode result)
        address-components  (into []
                                  (map parse-address-component)
                                  (.addressComponents result))
        formatted-address   (.formattedAddress result)
        geometry            (.geometry result)
        partial-match       (.partialMatch result)
        place-id            (.placeId result)
        postcode-localities (.postcodeLocalities result)
        types               (.types result)]
    (cond-> {}
      address-components  (assoc :address-components address-components)
      geometry            (assoc :geometry (parse-geometry geometry))
      types               (assoc :types (parse-address-types types))
      formatted-address   (assoc :formatted-address formatted-address)
      partial-match       (assoc :partial-match partial-match)
      place-id            (assoc :place-id place-id)
      postcode-localities (assoc :postcode-localities postcode-localities)
      plus-code           (assoc :plus-code (parse-plus-code plus-code)))))
