(ns closhare.consumer
  (:require [cheshire.core :as json]
            [clojure.string :as string]
            [com.stuartsierra.component :as component]
            [kithara.core :as k]
            [kithara.patterns.threads :as threads]
            [kithara.protocols :as protocols]))

(defmethod protocols/coerce-by-keyword :json
  [_ ^bytes body]
  (json/parse-string (String. body "UTF-8") true))

(defmethod protocols/coerce-by-keyword :edn
  [_ ^bytes body]
  (read-string (String. body "UTF-8")))

(defn uri->connection-map [s]
  (let [{:keys [host path userInfo]} (bean (java.net.URI. s))
        [username password] (when (seq userInfo)
                              (string/split userInfo #":"))]
    (cond-> {:host host
             :vhost (or (some-> (not-empty path) (string/replace #"^/" ""))
                        "/")}
      username (assoc :username username)
      password (assoc :password password))))

(defrecord Consumer [rabbitmq-uri handler-fn queue threads coerce]
  component/Lifecycle
  (start [this]
    (println ";; Starting consumer")
    (let [consumer (-> (partial handler-fn this)
                       (k/consumer {:as coerce})
                       (k/with-channel {:prefetch-count 256})
                       (k/with-queue queue)
                       (k/with-connection (uri->connection-map rabbitmq-uri)))]
      (component/start (cond-> consumer
                         threads (threads/with-threads threads)))))
  (stop [this]
    (println ";; Stopping consumer")
    this))

(defn consumer [rabbitmq-uri handler-fn queue & {:keys [threads coerce] :or {coerce :string}}]
  (Consumer. rabbitmq-uri handler-fn queue threads coerce))
