(ns edd.el.cmd
  (:require
   [clojure.tools.logging :as log]
   [lambda.util :as util]
   [edd.dal :as dal]
   [lambda.request :as request]
   [edd.response.cache :as response-cache]
   [malli.core :as m]
   [malli.error :as me]
   [edd.response.s3 :as s3-cache]
   [edd.ctx :as edd-ctx]
   [edd.el.ctx :as el-ctx]
   [edd.common :as common]
   [edd.el.event :as event]
   [edd.request-cache :as request-cache]
   [edd.schema.core :as edd-schema]
   [edd.el.query :as edd-client])
  (:import (clojure.lang ExceptionInfo)))

(def calc-service-query-url edd-client/calc-service-query-url)

(defn fetch-dependencies-for-command
  [ctx {:keys [cmd-id] :as cmd}]
  (log/infof "Query for dependency %s" cmd-id)
  (edd-client/fetch
   ctx
   (-> (edd-ctx/get-cmd ctx cmd-id)
       :deps)
   cmd))

(defn with-breadcrumbs
  [ctx resp]
  (let [parent-breadcrumb (or (get-in ctx [:breadcrumbs]) [])]
    (update resp :effects
            (fn [cmds]
              (vec
               (map-indexed
                (fn [i cmd]
                  (assoc cmd :breadcrumbs
                         (conj parent-breadcrumb i)))
                cmds))))))

(defn wrap-commands
  [ctx commands]
  (if-not (contains? (into [] commands) :commands)
    (into []
          (map
           (fn [cmd]
             (if-not (contains? cmd :commands)
               {:service  (:service-name ctx)
                :commands [cmd]}
               cmd))
           (if (map? commands)
             [commands]
             commands)))
    [commands]))

(defn clean-effects
  [ctx effects]
  (->> effects
       (remove nil?)
       (wrap-commands ctx)))

(defn handle-effects
  [ctx & {:keys [resp aggregate]}]
  (let [events (:events resp)
        ctx (el-ctx/put-aggregate ctx aggregate)
        effects (reduce
                 (fn [cmd fx-fn]
                   (let [resp (fx-fn ctx events)]
                     (into cmd
                           (if (map? resp)
                             [resp]
                             resp))))
                 []
                 (:fx ctx))
        effects (flatten effects)
        effects (clean-effects ctx effects)
        effects (map #(assoc %
                             :request-id (:request-id ctx)
                             :interaction-id (:interaction-id ctx)
                             :meta (:meta ctx {}))
                     effects)]

    (assoc resp :effects effects)))

(defn to-clean-vector
  [resp]
  (if (or (vector? resp)
          (list? resp))
    (remove nil?
            resp)
    (if resp
      [resp]
      [])))

(defn resp->add-meta-to-events
  [ctx {:keys [events] :as resp}]
  (assoc resp
         :events
         (map
          (fn [{:keys [error] :as %}]
            (if-not error
              (assoc % :meta (:meta ctx {}))
              %))
          events)))

(defn resp->add-user-to-events
  [{:keys [user]} resp]
  (if-not user
    resp
    (update resp :events
            (fn
              [events]
              (map #(assoc %
                           :user (:id user)
                           :role (:role user))
                   events)))))

(defn resolve-command-id-with-id-fn
  "Resolving command id. Taking into account override function of id.
  If id-fn returns null we fallback to command id. Override should
  be only used when it is impossible to create id on client. Like in case
  of import"
  [ctx cmd]
  (let [cmd-id (:cmd-id cmd)
        {:keys [id-fn]} (edd-ctx/get-cmd ctx cmd-id)]

    (if id-fn
      (let [new-id (id-fn ctx cmd)]
        (if new-id
          (assoc cmd :id new-id)
          cmd))
      cmd)))

(defn resp->assign-event-seq
  [ctx {:keys [events] :as resp}]
  (let [aggregate (el-ctx/get-aggregate ctx)
        last-sequence (:version aggregate 0)]
    (assoc resp
           :events (map-indexed (fn [idx event]
                                  (assoc event
                                         :event-seq
                                         (+ last-sequence idx 1)))
                                events))))

(defn verify-command-version
  [ctx cmd]
  (let [aggregate (el-ctx/get-aggregate ctx)
        {:keys [version]} cmd
        current-version (:version aggregate)]
    (cond
      (nil? version) ctx
      (not= version current-version) (throw (ex-info "Wrong version"
                                                     {:key   :concurrent-modification
                                                      :message "Version mismatch"
                                                      :state   {:current current-version
                                                                :version version}}))
      :else ctx)))

(defn invoke-handler
  "We add try catch here in order to parse
  all Exceptions thrown by handler in to data. Afterwards we want only ex-info"
  [handler cmd ctx]
  (try
    (handler ctx cmd)
    (catch Exception e
      (let [data (ex-data e)]
        (when data
          (throw e))
        (log/warn "Command handler failed" e)
        (throw (ex-info "Command handler failed"
                        {:message "Command handler failed"}))))))

(defn get-response-from-command-handler
  [ctx & {:keys [command-handler cmd]}]
  (verify-command-version ctx cmd)
  (let [events (->> ctx
                    (invoke-handler command-handler cmd)
                    (to-clean-vector)
                    (map #(assoc % :id (:id cmd)))
                    (remove nil?))
        response {:events     []
                  :identities []
                  :sequences  []}]
    (if (some :error events)
      (assoc response
             :error (vec
                     (filter :error events)))
      (reduce
       (fn [p event]
         (cond-> p
           (contains? event :identity) (update :identities conj event)
           (contains? event :sequence) (update :sequences conj event)
           (contains? event :event-id) (update :events conj event)))
       response
       events))))

(defn validate-single-command
  [ctx {:keys [cmd-id
               id] :as cmd}]
  (let [{:keys [consumes
                handler]} (edd-ctx/get-cmd ctx cmd-id)]
    (cond
      (not (m/validate (edd-schema/EddCoreCommand) cmd))
      (->> cmd
           (m/explain (edd-schema/EddCoreCommand))
           (me/humanize))

      (not handler)
      (str "Missing handler: " cmd-id)

      (not (m/validate consumes cmd))
      (->> cmd
           (m/explain consumes)
           (me/humanize))

      (not id) {:id ["missing required key"]}
      :else nil)))

(defn handle-command
  [ctx {:keys [cmd-id] :as cmd}]
  (log/info (:meta ctx))
  (util/d-time
   (str "handling-command: " cmd-id)
   (let [validation (validate-single-command ctx cmd)]
     (if validation
       {:error validation}
       (let [{:keys [handler]} (edd-ctx/get-cmd ctx cmd-id)
             dependencies (fetch-dependencies-for-command ctx cmd)
             ctx (merge dependencies ctx)
             {:keys [id] :as cmd} (resolve-command-id-with-id-fn ctx cmd)
             aggregate (common/get-by-id ctx {:id id})
             ctx (el-ctx/put-aggregate ctx aggregate)
             resp (->> (get-response-from-command-handler
                        ctx
                        :cmd cmd
                        :command-handler handler)
                       (resp->add-user-to-events ctx)
                       (resp->assign-event-seq ctx))
             resp (assoc resp :meta [{cmd-id {:id id}}])
             aggregate-schema (edd-ctx/get-service-schema ctx)
             aggregate (event/get-current-state ctx
                                                {:id id
                                                 :events (:events resp [])
                                                 :snapshot aggregate})]

         (when (and aggregate
                    (not
                     (m/validate aggregate-schema aggregate)))
           (throw (ex-info "Invalid aggregate state"
                           {:error (me/humanize
                                    (m/explain aggregate-schema aggregate))})))

         (let [resp (handle-effects ctx
                                    :resp resp
                                    :aggregate aggregate)

               resp (resp->add-meta-to-events ctx resp)]
           (request-cache/save-aggregate ctx aggregate)
           resp))))))

(def initial-response
  {:meta       []
   :events     []
   :effects    []
   :sequences  []
   :identities []})

(defn retry [f n]
  (try
    (f)
    (catch ExceptionInfo e
      (let [data (ex-data e)]
        (request-cache/clear)
        (if (and (= (get-in data [:key])
                    :concurrent-modification)
                 (not (zero? n)))
          (do
            (log/warnf e "Failed handling attempt: %d" n)
            (Thread/sleep (+ 1000 (rand-int 1000)))
            (retry f (dec n)))
          (throw e))))))

(defn- log-request [ctx body]
  (dal/log-request ctx body)
  ctx)

#_(defn resp->store-cache-partition
    [ctx resp]
    (response-cache/cache-response ctx resp))

#_(defn resp->cache-partitioned
    [ctx resp]
    (let [{:keys [effects]} resp
          partition-site (el-ctx/get-effect-partition-size ctx)]
      (if (< (count effects) partition-site)
        (resp->store-cache-partition ctx resp)
        (map-indexed
         (fn [idx %]
           (resp->store-cache-partition ctx (assoc resp :effects %
                                                   :idx idx)))
         (partition partition-site partition-site nil effects)))))

(defn store-results
  [ctx resp]
  (util/d-time
   "Storing results to dal"
   (dal/store-results (assoc ctx :resp resp))))

(defn process-commands
  [ctx {:keys [commands] :as body}]
  (let [ctx (-> ctx
                (assoc :commands commands
                       :breadcrumbs (or (get body :breadcrumbs)
                                        [0]))
                (s3-cache/register))]

    (log-request ctx body)

    (if-let [resp (:data (dal/get-command-response ctx))]
      resp
      (let [resp (loop [queue commands
                        resp initial-response
                        errors []]
                   (let [c (first queue)
                         r (when c
                             (handle-command ctx c))
                         resp (when r
                                (merge-with concat resp r))]
                     (cond
                       (nil? resp) resp
                       (:error r) (assoc initial-response
                                         :error (conj errors
                                                      (:error r)))
                       (:exception r) (assoc initial-response
                                             :exception (:exception r))
                       (seq
                        (rest queue)) (recur (rest queue)
                                             resp
                                             (conj errors nil))
                       :else resp)))
            resp (with-breadcrumbs ctx resp)
            resp (store-results ctx resp)]
        resp))))

(defn handle-commands
  [ctx body]
  (let [ctx (assoc ctx
                   :meta (merge
                          (:meta ctx {})
                          (:meta body {})))
        ctx (s3-cache/register ctx)]
    (try
      (retry #(process-commands ctx body)
             3)
      (catch ExceptionInfo e
        (dal/log-request-error ctx body (ex-data e))
        (throw (ex-info "All retrys exhosted" (ex-data e) e))))))

