(ns edd.el.event
  (:require
   [clojure.tools.logging :as log]
   [edd.dal :as dal]
   [edd.request-cache :as request-cache]
   [lambda.request :as request]
   [edd.view-store :as view-store]
   [lambda.util :as util]
   [lambda.ctx :as lambda-ctx]))

(defn apply-event
  [agr event func]
  (if func
    (assoc
     (apply func [agr event])
     :version (:event-seq event)
     :id (:id event))
    agr))

(defn create-aggregate
  [snapshot events apply-functions]
  (reduce
   (fn [agr event]
     (log/debug "Attempting to apply" event)
     (let [event-id (keyword (:event-id event))]

       (if (contains? apply-functions event-id)
         (apply-event
          agr
          event
          (event-id apply-functions))
         (assoc agr
                :version (:event-seq event)
                :id (:id event)))))
   snapshot
   events))

(defn apply-agg-filter
  [ctx aggregate]
  (reduce
   (fn [v f]
     (f (assoc
         ctx
         :agg v)))
   aggregate
   (get ctx :agg-filter [])))

(defn get-current-state
  [ctx {:keys [id
               snapshot
               events]}]
  {:pre [id events]}
  (log/infof "Calculating current state for: %s, applying %s events, current version %s"
             id
             (count events)
             (:version snapshot 0))

  (when (:error events)
    (throw (ex-info "Error fetching events" {:error events})))

  (if (> (count events) 0)
    (let [aggregate (create-aggregate snapshot
                                      events
                                      (:def-apply ctx))
          aggregate (apply-agg-filter ctx aggregate)]
      (log/infof "New version of aggregate %s is %s", id (:version aggregate 0))
      aggregate)
    (do
      (log/infof "No events applicable to %s, returning version %s", id (:version snapshot 0))
      snapshot)))

(defn fetch-snapshot
  [ctx id]
  (when-let [aggregate (view-store/get-snapshot (:view-store ctx) ctx id)]
    (log/info "Found snapshot: "
              (:id aggregate)
              ", "
              (:version aggregate))
    aggregate))

(defn get-events [ctx {:keys [id version]}]
  (dal/get-events (assoc ctx
                         :id id
                         :version version)))

(defn get-by-id
  [ctx id]
  {:pre [id]}
  (if-let [aggregate (request-cache/get-aggregate ctx id)]
    (do
      (log/info (str "Using cached snapshot: "
                     (:id aggregate)
                     ", "
                     (:version aggregate)))
      aggregate)
    (let [{:keys [version]
           :as snapshot} (fetch-snapshot ctx id)
          events (get-events ctx
                             {:id id
                              :version version})]
      (get-current-state ctx {:id id
                              :snapshot snapshot
                              :events events}))))

(defn save-aggregate
  [ctx aggregate]
  (when aggregate
    (util/d-time
     (str "Updating aggregate: "
          (lambda-ctx/realm ctx)
          ", "
          (:id aggregate)
          ", "
          (:version aggregate))
     (view-store/save-aggregate (:view-store ctx) ctx aggregate)))
  ctx)

(defn handle-event
  [ctx body]
  (let [apply-request (:apply body)
        meta (merge (:meta ctx)
                    (:meta body))
        ctx (assoc ctx :meta meta)
        realm (:realm meta)
        agg-id (:aggregate-id apply-request)]

    (util/d-time
     (str "handling-apply: " realm " " (:aggregate-id apply))
     (let [applied (get-in @request/*request* [:applied realm agg-id])]
       (when-not applied
         (save-aggregate ctx (get-by-id ctx agg-id))
         (swap! request/*request*
                #(assoc-in % [:applied realm agg-id] {:apply true}))))))
  {:apply true})
