(ns edd.el.query
  (:require [clojure.tools.logging :as log]
            [malli.core :as m]
            [lambda.http-client :as http-client]
            [edd.ctx :as edd-ctx]
            [edd.schema :as schema]
            [aws.aws :as aws]
            [lambda.util :as util]))

(defn- validate-response-schema-setting [ctx]
  (let [setting (or (:response-schema-validation ctx)
                    (:response-schema-validation (:meta ctx)))]
    (assert #{nil :log-on-error :throw-on-error} setting)
    setting))

(defn- validate-response-schema? [ctx]
  (contains? #{:log-on-error :throw-on-error}
             (validate-response-schema-setting ctx)))

(defn- maybe-validate-response [ctx produces response]
  (when (and produces
             (validate-response-schema? ctx))
    (let [wrapped {:result response}]
      (when-not (m/validate produces wrapped)
        (let [error (schema/explain-error produces wrapped)]
          (condp = (validate-response-schema-setting ctx)
            :throw-on-error
            (throw (ex-info "Invalid response" {:error    error
                                                :response wrapped}))
            :log-on-error
            (log/warnf "Invalid response %s" (pr-str {:error error
                                                      :response wrapped}))))))))
(declare fetch)

(defn handle-query
  [ctx body]
  (let [query (:query body)
        query-id (keyword (:query-id query))
        {:keys [consumes
                handler
                produces]} (get-in ctx [:edd-core :queries query-id])]

    (log/debug "Handling query" query-id)
    (when-not handler
      (throw (ex-info "No handler found"
                      {:error    "No handler found"
                       :query-id query-id})))
    (when-not (m/validate consumes query)
      (throw (ex-info "Invalid request"
                      {:error (schema/explain-error consumes query)})))
    (let [{:keys [deps]} (edd-ctx/get-query ctx query-id)
          deps-value (fetch ctx deps query)
          ctx (merge ctx deps-value)
          resp (util/d-time
                (str "handling-query: " query-id)
                (handler ctx query))]
      (log/debug "Query response" resp)
      (maybe-validate-response ctx produces resp)
      resp)))

(defn calc-service-query-url
  [service]
  (str "https://api."
       (util/get-env "PrivateHostedZoneName")
       "/private/prod/"
       (name service)
       "/query"))

(defn call-query-fn
  [_ cmd query-fn deps]
  (query-fn deps cmd))

(defn resolve-remote-dependency
  [ctx cmd {:keys [service query]} deps]
  (log/info "Resolving remote dependency: " service (or (:cmd-id cmd)
                                                        (:query-id cmd)))

  (let [query-fn query
        service-name (:service-name ctx)
        url (calc-service-query-url
             service)
        token (aws/get-token ctx)
        resolved-query (call-query-fn ctx cmd query-fn deps)
        response (when (or
                        (get-in resolved-query [:query-id])
                        (get-in resolved-query [:query :query-id]))
                   (http-client/retry-n
                    #(util/http-request
                      url
                      (http-client/request->with-timeouts
                       %
                       {:method :post
                        :body    (util/to-json
                                  {:query          resolved-query
                                   :meta           (:meta ctx)
                                   :request-id     (:request-id ctx)
                                   :interaction-id (:interaction-id ctx)})
                        :headers {"Content-Type"    "application/json"
                                  "X-Authorization" token}}
                       :idle-timeout 10000))
                    :meta {:to-service   service
                           :from-service service-name
                           :query-id     (:query-id resolved-query)}))]
    (when (:error response)
      (throw (ex-info (str "Error fetching dependency" service)
                      {:error {:to-service   service
                               :from-service service-name
                               :query-id     (:query-id resolved-query)
                               :message      (:error response)}})))
    (when (:error (get response :body))
      (throw (ex-info (str "Error response from service " service)
                      {:error {:to-service   service
                               :from-service service-name
                               :query-id     (:query-id resolved-query)
                               :message      {:response     (get response :body)
                                              :error-source service}}})))
    (if (> (:status response 0) 299)
      (throw (ex-info (str "Deps request error for " service)
                      {:error {:to-service   service
                               :from-service service-name
                               :service      service
                               :query-id     (:query-id resolved-query)
                               :status       (:status response)
                               :message      (str "Response status:" (:status response))}}))
      (get-in response [:body :result]))))

(defn resolve-local-dependency
  [ctx cmd query-fn deps]
  (log/debug "Resolving local dependency")
  (let [query (call-query-fn ctx cmd query-fn deps)]
    (when query
      (let [resp (handle-query ctx {:query query})]
        (if (:error resp)
          (throw (ex-info "Failed to resolve local deps" {:error resp}))
          resp)))))

(defn fetch
  [ctx deps request]
  (let [deps (if (vector? deps)
               (partition 2 deps)
               deps)
        dps-value (reduce
                   (fn [p [key req]]
                     (let [dep-value
                           (try (if (:service req)
                                  (resolve-remote-dependency
                                   ctx
                                   request
                                   req
                                   p)
                                  (resolve-local-dependency
                                   ctx
                                   request
                                   req
                                   p))
                                (catch AssertionError e
                                  (log/warn "Assertion error for deps " key)
                                  nil))]
                       (if dep-value
                         (assoc p key dep-value)
                         p)))
                   {}
                   deps)]
    dps-value))


