(ns edd.view-store.searchable.opensearch
  (:require [clojure.tools.logging :as log]
            [malli.core :as m]
            [lambda.util :as util]
            [lambda.ctx :as lambda-ctx]
            [lambda.elastic :as el]
            [clojure.string :as string]
            [edd.view-store :as view-store]
            [edd.view-store.searchable.common :as searchable-common]
            [edd.view-store.searchable :as searchable-view-store]))

(def config-key :opensearch->-config)

(defn set-config [ctx config]
  (assoc ctx
         config-key
         config))

(defn get-config [ctx]
  (get ctx
       config-key))

(defn make-index-name
  [realm-name service-name]
  (str (name realm-name)
       "_"
       (string/replace
        (name service-name)
        "-"
        "_")))

(defn ctx->index-name
  [{:keys [service-name
           index-name]
    :as ctx}]
  (make-index-name
   (lambda-ctx/realm ctx)
   (or index-name
       service-name)))

(defn field+keyword
  [field]
  (str (name field) ".keyword"))

(defn trim-string [v]
  (if (string? v)
    (string/trim v)
    v))

(def op->filter-builder
  {:and      (fn [op->fn & filter-spec]
               {:bool
                {:filter (mapv #(searchable-common/parse op->fn %) filter-spec)}})
   :or       (fn [op->fn & filter-spec]
               {:bool
                {:should               (mapv #(searchable-common/parse op->fn %) filter-spec)
                 :minimum_should_match 1}})
   :eq       (fn [_ & [a b]]
               {:term
                {(field+keyword a) (trim-string b)}})
   :=        (fn [_ & [a b]]
               {:term
                {(name a) (trim-string b)}})
   :wildcard (fn [_ & [a b]]
               {:bool
                {:should
                 [{:wildcard
                   {(str (name a)) {:value (str "*" (trim-string b) "*")}}}
                  {:match_phrase
                   {(str (name a)) (str (trim-string b))}}]}})
   :not      (fn [op->fn & filter-spec]
               {:bool
                {:must_not (searchable-common/parse op->fn filter-spec)}})
   :in       (fn [_ & [a b]]
               {:terms
                {(field+keyword a) b}})
   :exists   (fn [_ & [a _]]
               {:exists
                {:field (name a)}})
   :lte      (fn [_ & [a b]]
               {:range
                {(name a) {:lte b}}})
   :gte      (fn [_ & [a b]]
               {:range
                {(name a) {:gte b}}})
   :nested   (fn [op->fn path & filter-spec]
               {:bool
                {:must [{:nested {:path  (name path)
                                  :query (mapv #(searchable-common/parse op->fn %) filter-spec)}}]}})})

(defn search-with-filter
  [filter q]
  (let [[_fields-key fields _value-key value] (:search q)
        search (mapv
                (fn [p]
                  {:bool
                   {:should               [{:match
                                            {(str (name p))
                                             {:query value
                                              :boost 2}}}
                                           {:wildcard
                                            {(str (name p)) {:value (str "*" (trim-string value) "*")}}}]
                    :minimum_should_match 1}})
                fields)]

    (-> filter
        (assoc-in [:query :bool :should] search)
        (assoc-in [:query :bool :minimum_should_match] 1))))

(defn form-sorting
  [sort]
  (map
   (fn [[a b]]
     (case (keyword b)
       :asc {(field+keyword a) {:order "asc"}}
       :desc {(field+keyword a) {:order "desc"}}
       :asc-number {(str (name a) ".number") {:order "asc"}}
       :desc-number {(str (name a) ".number") {:order "desc"}}
       :asc-date {(name a) {:order "asc"}}
       :desc-date {(name a) {:order "desc"}}))
   (partition 2 sort)))

(defn create-elastic-query
  [q]
  (cond-> {}
    (:filter q) (merge {:query {:bool {:filter (searchable-common/parse op->filter-builder (:filter q))}}})
    (:search q) (search-with-filter q)
    (:select q) (assoc :_source (mapv name (:select q)))
    (:sort q) (assoc :sort (form-sorting (:sort q)))))

(defn advanced-direct-search
  [ctx elastic-query]
  (let [json-query (util/to-json elastic-query)
        index-name (ctx->index-name ctx)
        {:keys [error] :as body} (el/query
                                  {:config (get-config ctx)
                                   :method "GET"
                                   :path   (str "/" index-name "/_search")
                                   :body   json-query})
        total (get-in body [:hits :total :value])]

    (when error
      (throw (ex-info "Elastic query failed" error)))
    (log/debug "Elastic query")
    (log/debug json-query)
    (log/debug body)
    {:total total
     :from  (get elastic-query :from 0)
     :size  (get elastic-query :size searchable-common/default-size)
     :hits  (mapv
             :_source
             (get-in body [:hits :hits] []))}))

(defn impl->advanced-search
  [ctx query]
  (let [elastic-query (-> (create-elastic-query query)
                          (assoc
                           :from (get query :from 0)
                           :size (get query :size searchable-common/default-size)))]
    (advanced-direct-search ctx elastic-query)))

(defn flatten-paths
  ([m separator]
   (flatten-paths m separator []))
  ([m separator path]
   (->> (map (fn [[k v]]
               (if (and (map? v) (not-empty v))
                 (flatten-paths v separator (conj path k))
                 [(->> (conj path k)
                       (map name)
                       (string/join separator)
                       keyword) v]))
             m)
        (into {}))))

(defn create-simple-query
  [query]
  {:pre [query]}
  (util/to-json
   {:size  600
    :query {:bool
            {:must (mapv
                    (fn [[field value]]
                      {:term {(field+keyword field) value}})
                    (seq (flatten-paths query ".")))}}}))

(defn impl->simple-search
  [ctx query]
  (log/debug "Executing simple search" query)
  (let [index-name (ctx->index-name ctx)
        param (dissoc query :query-id)
        body (util/d-time
              "Doing elastic search (Simple-search)"
              (el/query
               {:config (get-config ctx)
                :method         "POST"
                :path           (str "/" index-name "/_search")
                :body           (create-simple-query param)
                :elastic-search (get-config ctx)
                :aws            (:aws ctx)}))]
    (mapv
     :_source
     (get-in body [:hits :hits] []))))

(defn impl->save-snapshot
  [ctx aggregate]
  (log/info "Updating aggregate " (lambda-ctx/realm ctx) (:id aggregate) (:version aggregate))
  (let [index-name (ctx->index-name ctx)
        {:keys [error]} (el/query
                         {:config (get-config ctx)
                          :method "POST"
                          :path   (str "/" index-name "/_doc/" (:id aggregate))
                          :body   (util/to-json aggregate)
                          :aws    (:aws ctx)})]
    (if error
      (throw (ex-info "Could not store aggregate" {:error error}))
      ctx)))

(defn impl->get-snapshot
  [ctx id]
  (log/info "Fetching snapshot aggregate" (lambda-ctx/realm ctx) id)
  (util/d-time
   (str "Fetching snapshot aggregate in realm: " (lambda-ctx/realm ctx)
        ", id: " id)
   (let [index-name (ctx->index-name ctx)
         {:keys [error] :as body} (el/query
                                   {:config (get-config ctx)
                                    :method "GET"
                                    :path   (str "/" index-name "/_doc/" id)
                                    :aws    (:aws ctx)}
                                   :ignored-status 404)]
     (if error
       (throw (ex-info "Failed to fetch snapshot" error))
       (:_source body)))))

(defrecord OpenSearchViewStore [config]
  view-store/ViewStore
  (init [_this ctx]
    (log/info "Intializing OpenSearch view-store")
    ctx)
  (get-snapshot [_this ctx query]
    (impl->get-snapshot (set-config ctx config)
                        query))
  (save-aggregate [_this ctx aggregate]
    (impl->save-snapshot (set-config ctx config)
                         aggregate))

  searchable-view-store/SearchableViewStore
  (advanced-search [_this ctx query]
    (impl->advanced-search (set-config ctx config)
                           query))
  (simple-search [_this ctx query]
    (impl->simple-search (set-config ctx config)
                         query)))

(def default-endpoint "127.0.0.1:9200")

(defn register
  [ctx & [config]]
  (view-store/register
   ctx
   {:implementation-class OpenSearchViewStore
    :config (or config {})
    :config-default {:scheme (util/get-env "IndexDomainScheme" "https")
                     :url    (util/get-env "IndexDomainEndpoint" default-endpoint)}
    :config-schema (m/schema
                    [:map
                     [:scheme string?]
                     [:url string?]])}))
