(ns lambda.runtime
  (:require
   [clojure.tools.logging :as log]
   [lambda.core :as lambda-core]
   [lambda.ctx :as lambda-ctx]
   [lambda.request :as request]
   [lambda.test.fixture.core :as lambda-fixture-core]
   [lambda.util :as util]
   [lambda.uuid :as uuid]
   [malli.core :as m]
   [malli.error :as me]
   [sdk.aws.common :as aws-common]))

#_(let  [filter (first filters)
         remaining-filters (rest filters)
         remaining-filters (if (seq remaining-filters)
                             remaining-filters
                             nil)
         filter-chain (if remaining-filters
                        #(apply-filters % remaining-filters {} nil)
                        nil)]
    (if (apply apply-filters nil nil nil nil)
      (lambda-core/do-handle
       handler
       ctx
       filter-chain)
      (if remaining-filters
        (apply-filters ctx remaining-filters nil nil)
        (throw (ex-info "Last filter (Handler) should always be applicatble"
                        {:remaining remaining-filters})))))

(def handler-missing-exception-message
  "Handler is required")

(defn apply-filters
  [ctx filters handler request]
  (when-not handler
    (throw (ex-info handler-missing-exception-message
                    {:exception handler-missing-exception-message})))
  (if (seq filters)
    (loop [reversed (reverse filters)
           ; Default filter chain is to invoke handler
           filter-chain (reify
                          lambda-core/FilterChain
                          (continue-filter [_this ctx request]
                            (log/info "Filtering complete. Invoking hanler")
                            (lambda-core/do-handle
                             handler
                             ctx
                             request)))]

      (if-let [filter (first reversed)]
        (recur (rest reversed)
               (reify
                 lambda-core/FilterChain
                 (continue-filter [_this ctx request]
                   (log/info
                    (format "Applying filter: %s"
                            (type filter)))
                   (lambda-core/do-filter
                    filter
                    ctx
                    request
                    filter-chain))))
        ; When no more filters remaining we start it all
        (lambda-core/continue-filter
         filter-chain
         ctx
         request)))

    (do
      (log/info "No filters found so invoking handler")
      (lambda-core/do-handle
       handler
       ctx
       request))))

(defn invoke-handler
  [{:keys [body] :as ctx} & {:keys [handler]}]
  (util/d-time
   "time-invoke-handler"
   (handler ctx body)))

(defn get-loop
  "Extracting lambda looping as infinite loop to be able to mock it"
  []
  (range))

(defn ctx->load-config
  [ctx]
  (let [config-file (get-in ctx
                            [:edd :config :secrets-file])]
    (if config-file
      (do
        (log/info (format "Loading config file: %s" config-file))
        (merge ctx
               (util/load-config config-file)))
      (do
        (log/warn "Config file is missing fron ctx configuration")
        ctx))))

(defn impl->init-filter
  [ctx filter]
  (log/info (format "Initializing filter: %s" filter))
  (lambda-core/init-filter filter ctx))

(def invalid-runtime-configuration-message
  "Invalid runtime configuration")

(defn impl->init-runtime
  [ctx {:keys [handler filters]
        :as config}]
  (util/d-time "Initializing core runtime"
               (let [runtime-schema (m/schema
                                     [:map
                                      [:handler]])
                     _ (when-not (m/validate runtime-schema
                                             config)
                         (throw (ex-info invalid-runtime-configuration-message
                                         {:exception
                                          {:message invalid-runtime-configuration-message
                                           :data (-> (m/explain runtime-schema config)
                                                     (me/humanize))}})))
                     ctx (ctx->load-config ctx)
                     ctx (reduce
                          (fn [ctx v]
                            (impl->init-filter ctx v))
                          ctx
                          filters)
                     ctx (lambda-core/init-handler handler ctx)
                     ctx (assoc ctx :application-scoped-cache (atom {}))]
                 (assoc ctx :runtime-initialized true))))

(def original-request-key :original-request)
(defn runtime->get-original-request
  [ctx]
  (get original-request-key ctx))

(defn impl->run-request
  [ctx {:keys [request
               config
               invocation-id]}]

  #_(when-not (:filter-initialized ctx)
      (throw (ex-info "Filters not initialized"
                      {:message "Runtime should initize all filters"})))
  (let [{:keys [handler
                filters]} config
        invocation-id (uuid/parse invocation-id)]
    (binding [request/*request* (atom {:mdc {:invocation-id invocation-id}})]
      (try
        (let [ctx (lambda-ctx/init ctx)
              request (if (string? request)
                        (util/to-edn request)
                        request)
              ctx (assoc ctx
                         original-request-key request
                         :invocation-id invocation-id)
              response (apply-filters ctx filters handler request)]
          (doall response))
        (catch Exception e
          (log/error e "Error processing request")
          (util/exception->response e))
        (catch AssertionError e
          (log/error e "Assertion processing request")
          (util/exception->response e))))))

(def created-date "20200426T061823Z")

(defn memory-runtime->run-request
  [ctx config invocation-id request]
  (util/d-time
   (str "Handling request: " invocation-id)
   (let [response (with-redefs [util/get-env
                                (partial lambda-fixture-core/get-env-mock (:env config {}))]

                    (impl->run-request
                     ctx
                     {:request request
                      :config config
                      :invocation-id invocation-id}))]

     response)))

(deftype MemoryRuntime [config ^clojure.lang.Atom state]
  lambda-core/LambdaRuntime
  (init-runtime [_this ctx]
    (log/info "Initializing MemoryRuntime")
    (reset! state 0)
    (impl->init-runtime ctx config))
  (start-runtime [_this _ctx]
    (log/info "Starting MemoryRuntime"))
  (run-request [_this ctx request]
    (let [invocation-id (:invocation-id
                         ctx
                         (uuid/from-number @state))]
      (swap! state inc)
      (memory-runtime->run-request ctx config invocation-id request))))

(deftype PassTroughMiddleware []
  lambda-core/LambdaMiddlware
  (init-middleware [_this ctx]
    ctx)
  (do-process [_this ctx request middleware-chain]
    (lambda-core/continue-process middleware-chain ctx request)))
