(ns lambda.util
  (:require [jsonista.core :as json]
            [clojure.tools.logging :as log]
            [clojure.java.io :as io]
            [clojure.string :as string]
            [clojure.set :as clojure-set]
            [lambda.aes :as aes]
            [java-http-clj.core :as http]
            [clojure.walk :as walk])
  (:import (java.time OffsetDateTime)
           (java.time.format DateTimeFormatter)
           (java.io File BufferedReader)
           (java.util UUID Date Base64)
           (javax.crypto Mac)
           (javax.crypto.spec SecretKeySpec)
           (com.fasterxml.jackson.core JsonGenerator)
           (clojure.lang Keyword)
           (com.fasterxml.jackson.databind ObjectMapper)
           (java.nio.charset Charset)
           (java.security MessageDigest)
           (java.math BigInteger)
           (java.net URLDecoder)
           (java.net URLEncoder)
           (org.slf4j MDC)))

(def offset-date-time-format "yyyy-MM-dd'T'HH:mm:ss.SSSXXX")

(def ^ObjectMapper json-mapper
  (json/object-mapper
   {:decode-key-fn true
    :decode-fn
    (fn [v]
      (condp instance? v
        String (case (first v)
                 \: (if (string/starts-with? v "::")
                      (subs v 1)
                      (keyword (subs v 1)))
                 \# (if (string/starts-with? v "##")
                      (subs v 1)
                      (UUID/fromString (subs v 1)))
                 v)
        v))
    :encoders      {String         (fn [^String v ^JsonGenerator jg]
                                     (cond
                                       (string/starts-with? v ":")
                                       (.writeString jg (str ":" v))
                                       (string/starts-with? v "#")
                                       (.writeString jg (str "#" v))
                                       :else (.writeString jg v)))
                    BufferedReader (fn [^BufferedReader _v ^JsonGenerator jg]
                                     (.writeString jg "BufferedReader"))
                    UUID           (fn [^UUID v ^JsonGenerator jg]
                                     (.writeString jg (str "#" v)))
                    Keyword        (fn [^Keyword v ^JsonGenerator jg]
                                     (.writeString jg (str ":" (name v))))}}))

(defn date-time
  ([] (OffsetDateTime/now))
  ([^String value] (OffsetDateTime/parse value)))

(defn date->string
  ([] (.format (date-time) (DateTimeFormatter/ofPattern offset-date-time-format)))
  ([^OffsetDateTime date] (.format date (DateTimeFormatter/ofPattern offset-date-time-format))))

(defn get-current-time-ms
  []
  (System/currentTimeMillis))

(defn is-in-past
  [^Date date]
  (.before date (new Date)))

(defn to-edn
  [json]
  (json/read-value json json-mapper))

(defn to-json
  [edn]
  (json/write-value-as-string edn json-mapper))

(defn wrap-body [request]
  (cond
    (:form-params request) request
    (string? request) {:body request}
    :else {:body (to-json request)}))

(defn url-encode
  [s]
  (URLEncoder/encode (str s) "utf8"))

(defn url-decode
  [s]
  (URLDecoder/decode (str s) "utf8"))

(defn nested-param
  "Source: https://github.com/http-kit/http-kit"
  [params]
  (walk/prewalk (fn [d]
                  (if (and (vector? d) (map? (second d)))
                    (let [[fk m] d]
                      (reduce (fn [m [sk v]]
                                (assoc m (str (name fk) \[ (name sk) \]) v))
                              {} m))
                    d))
                params))

(defn query-string
  "Returns URL-encoded query string for given params map.
   Source: https://github.com/http-kit/http-kit"
  [m]
  (let [m (nested-param m)
        param (fn [k v]  (str (url-encode (name k)) "=" (url-encode v)))
        join  (fn [strs] (string/join "&" strs))]
    (join (for [[k v] m] (if (sequential? v)
                           (join (map (partial param k) (or (seq v) [""])))
                           (param k v))))))

(defn request
  [{:keys [as]
    :as req} & _rest]
  (let [opts {:as (cond
                    (and as
                         (= as :stream)) :input-stream
                    :else as)}
        req (clojure-set/rename-keys req {:url :uri})
        req (cond-> req
              (:headers req) (update-in [:headers]
                                        #(dissoc % "Host"))
              (:query-params req) (update-in [:uri]
                                             #(str %
                                                   "?"
                                                   (query-string (:query-params req))))
              (:form-params req) (assoc-in [:headers "Content-Type"] "application/x-www-form-urlencoded")
              (:form-params req) (update-in [:body]
                                            #(query-string (:form-params req))))
        resp (http/send
              req
              opts)
        resp (update resp :headers walk/keywordize-keys)]
    resp))

(defn http-request
  [url req & {:keys [raw]}]
  (log/debug url req)
  (let [resp (request (assoc req
                             :url url))]
    (log/debug "Response" resp)
    (if raw
      resp
      (assoc resp
             :body (to-edn (:body resp))))))

(defn http-get
  [url request & {:keys [raw]
                  :or {raw false}}]
  (http-request url (assoc request
                           :method :get) :raw raw))

(defn http-delete
  [url request & {:keys [raw]
                  :or {raw false}}]
  (http-request url (assoc request
                           :method :delete)
                :raw raw))

(defn http-put
  [url request & {:keys [raw]
                  :or {raw false}}]
  (log/debug url request)
  (http-request url (assoc (wrap-body request)
                           :method :put) :raw raw))

(defn http-post
  [url request & {:keys [raw]
                  :or {raw false}}]
  (http-request url (assoc (wrap-body request)
                           :method :post) :raw raw))

(defn get-env
  [name & [default]]
  (get (System/getenv) name default))

(defn get-property
  [name & [default]]
  (get (System/getProperties) name default))

(defn escape
  [value]
  (string/replace value "\"" "\\\""))

(defn decrypt
  [body ^String name]
  (log/debug "Decrypting")
  (let [context (get-env "ConfigurationContext")]
    (if (and context
             (.contains name "secret"))
      (let [context (string/split context #":")
            iv (first context)
            key (second context)]
        (aes/decrypt (string/replace body #"\n" "")
                     key
                     iv))
      body)))

(defn load-config
  [name]
  (log/debug "Loading config name:" name)
  (let [file (io/as-file name)
        config (to-edn
                (if (.exists ^File file)
                  (do
                    (log/debug "Loading from file config:" name)
                    (-> file
                        (slurp)
                        (decrypt name)))
                  (do
                    (log/debug "Loading config from classpath:" name)
                    (-> (slurp (io/resource name))
                        (decrypt name)))))
        env-config (to-edn
                    (get-env "CustomConfig" "{}"))]
    (merge config env-config)))

(defn base64encode
  [^String to-encode]
  (.encodeToString (Base64/getEncoder)
                   (.getBytes to-encode "UTF-8")))

(defn bytes->base64encode
  [to-encode]
  (.encodeToString (Base64/getEncoder)
                   to-encode))

(defn base64decode
  [^String to-decode]
  (String. (.decode
            (Base64/getDecoder)
            to-decode) "UTF-8"))

(defn base64URLdecode
  [^String to-decode]
  (String. (.decode
            (Base64/getUrlDecoder)
            to-decode) "UTF-8"))

(def ^:dynamic *cache*)

(defn hmac-sha256
  [^String secret ^String message]
  (let [mac (Mac/getInstance "HmacSHA256")
        secret-key-spec (new SecretKeySpec
                             (.getBytes secret "UTF-8")
                             "HmacSHA256")
        message-bytes (.getBytes message "UTF-8")]
    (.init mac secret-key-spec)
    (->> message-bytes
         (.doFinal mac)
         (.encodeToString (Base64/getEncoder)))))

(defn url-encode
  [^String message]
  (let [^Charset charset (Charset/forName "UTF-8")]
    (URLEncoder/encode message charset)))

(def ^:dynamic *d-time* (atom {:format :text}))

(defmacro d-time
  "Evaluates expr and logs time it took.  Returns the value of
 expr."
  {:added "1.0"}
  [message & expr]
  `(do
     (log/info (case (:format @*d-time*)
                 :json {:message (str "START " ~message)}
                 :text (str "START " ~message)))

     (let [prefix-before# (MDC/get "prefix")
           put# (MDC/put "prefix" (str prefix-before# "  "))
           start# (. System (nanoTime))
           mem# (-> (- (.totalMemory (Runtime/getRuntime))
                       (.freeMemory (Runtime/getRuntime)))
                    (/ 1024)
                    (/ 1024)
                    (int))
           ret# (do ~@expr)]

       (MDC/put "prefix" prefix-before#)
       (log/info  (case (:format @*d-time*)
                    :json {:type    :time
                           :message (str "END " ~message)
                           :elapsed (/ (double (- (. System (nanoTime)) start#)) 1000000.0)
                           :memory  (str mem# " -> " (-> (- (.totalMemory (Runtime/getRuntime))
                                                            (.freeMemory (Runtime/getRuntime)))
                                                         (/ 1024)
                                                         (/ 1024)
                                                         (int)))
                           :unit    "msec"}
                    :text (str "ENDED " ~message
                               "; "
                               "Elapsed: " (/ (double (- (. System (nanoTime)) start#)) 1000000.0) " msec; "
                               "Memory: " (str mem# " -> " (-> (- (.totalMemory (Runtime/getRuntime))
                                                                  (.freeMemory (Runtime/getRuntime)))
                                                               (/ 1024)
                                                               (/ 1024)
                                                               (int))) " Mb")))
       ret#)))

(defn exception->response
  [e]
  (let [data (ex-data e)]
    (if data
      (if (:exception data)
        data
        {:exception data})
      {:exception (try (.getMessage e)
                       (catch IllegalArgumentException e
                         (log/error e)
                         e))})))

(defn try->data
  [handler]
  (try (handler)
       (catch Exception e
         (exception->response e))))

(defn try->error
  [handler]
  (try (handler)
       (catch Exception e
         (clojure-set/rename-keys (exception->response e)
                                  {:exception :error}))))

(defn fix-keys
  "This is used to represent as close as possible when we store
  to external storage as JSON. Because we are using :keywordize keys
  for convenience. Problem is when map keys are in aggregate stored as strings.
  Then when they are being read from storage they are being keywordized.
  This is affecting when we are caching aggregate between calls because in
  this case cached aggregate would not represent real aggregate without cache.
  Other scenario is for tests because in tests we would get aggregate with string
  keys while in real scenario we would have keys as keywords."
  [val]
  (-> val
      (to-json)
      (to-edn)))

(defn log-startup
  []
  (let [startup-milis (Long/parseLong
                       (str
                        (get-property "edd.startup-milis" 0)))]
    (when (not= startup-milis 0)
      (log/info "Server started: " (- (System/currentTimeMillis)
                                      startup-milis)))))

(defn md5hadh [^String s]
  (let [algorithm (MessageDigest/getInstance "MD5")
        raw (.digest algorithm (.getBytes s))]
    (format "%032x" (BigInteger. 1 raw))))

(defn string->md5base64 [^String s]
  (let [algorithm (MessageDigest/getInstance "MD5")
        raw (.digest algorithm (.getBytes s))]
    (bytes->base64encode raw)))

(defn slurp-bytes
  "Slurp the bytes from a slurpable thing"
  [^String x]
  (with-open [out (java.io.ByteArrayOutputStream.)]
    (with-open [file (clojure.java.io/input-stream x)]
      (clojure.java.io/copy file out))
    (.toByteArray out)))

(defn path->md5base64 [^String path]
  (let [algorithm (MessageDigest/getInstance "MD5")
        raw (.digest algorithm  (slurp-bytes path))]
    (bytes->base64encode raw)))

(defn sha256 [^String s]
  (let [algorithm (MessageDigest/getInstance "MD5")
        raw (.digest algorithm (.getBytes s))]
    (format "%032x" (BigInteger. 1 raw))))

(defn hex-str-to-bit-str
  [hex]
  (case hex
    "0" "0000"
    "1" "0001"
    "2" "0010"
    "3" "0011"
    "4" "0100"
    "5" "0101"
    "6" "0110"
    "7" "0111"
    "8" "1000"
    "9" "1001"
    "a" "1010"
    "b" "1011"
    "c" "1100"
    "d" "1101"
    "e" "1110"
    "f" "1111"))

(defn url->map
  [url]
  (let [parts (string/split url #"&")
        parts (map #(string/split % #"=") parts)]
    (reduce
     (fn [p [entry-key entry-value]]
       (assoc p entry-key entry-value))
     {}
     parts)))
