(ns edd.view-store.searchable.memory
  (:require
   [clojure.data :as clojure-data]
   [clojure.string :as string]
   [clojure.tools.logging :as log]
   [edd.view-store :as view-store]
   [edd.view-store.searchable :as searchable-view-store]
   [edd.view-store.searchable.common :as searchable-common]
   [lambda.ctx :as lambda-ctx]
   [lambda.util :as util]
   [malli.core :as m]))

(def default-elastic-store {:aggregate-store []})

(defn to-keywords
  [a]
  (cond
    (keyword? a) (to-keywords (name a))
    (vector? a) (vec
                 (reduce
                  (fn [v p]
                    (concat v (to-keywords p)))
                  []
                  a))
    :else (map
           keyword
           (remove
            empty?
            (string/split a #"\.")))))

(defn and-fn
  [ctx & r]
  (fn [%]
    (let [result (every?
                  (fn [p]
                    (let [rest-fn (searchable-common/parse ctx p)]
                      (rest-fn %)))
                  r)]
      result)))

(defn or-fn
  [mock & r]
  (fn [%]
    (let [result (some
                  (fn [p]
                    (let [rest-fn (searchable-common/parse mock p)]
                      (rest-fn %)))
                  r)]
      (if result
        result
        false))))

(defn eq-fn
  [_ & [a b]]
  (fn [%]
    (let [keys (to-keywords a)
          response (= (get-in % keys)
                      (case b
                        string? (string/trim b)
                        b))]

      response)))

(defn not-fn
  [mock & [rest]]
  (fn [%]
    (not
     (apply
      (searchable-common/parse mock rest) [%]))))

(defn in-fn
  [_ key & [values]]
  (fn [p]
    (let [keys (to-keywords key)
          value (get-in p keys)]
      (if (some
           #(= % value)
           values)
        true
        false))))
(defn exists-fn
  [_ key & [_values]]
  (fn [p]
    (let [keys (to-keywords key)]
      (not= (get-in p keys :nil) :nil))))

(def mock
  {:and    and-fn
   :or     or-fn
   :exists exists-fn
   :eq     eq-fn
   :not    not-fn
   :in     in-fn})

(defn search-fn
  [q p]
  (let [[_fields-key fields _value-key value] (:search q)]
    (if (some
         #(let [v (get-in p (to-keywords %) "")]
            (.contains v value))
         fields)
      true
      false)))

(defn field-to-kw-list
  [p]
  (cond
    (string? p) (map
                 keyword
                 (string/split p #"\."))
    (keyword? p) (map
                  keyword
                  (string/split (name p) #"\."))))

(defn select-fn
  [q %]
  (reduce
   (fn [v p]
     (assoc-in v p
               (get-in % p)))
   {}
   (map
    field-to-kw-list
    (get q :select []))))

(defn compare-as-number
  [a b]
  (let [num_a (if (number? a)
                a
                (Integer/parseInt a))
        num_b (if (number? b)
                b
                (Integer/parseInt b))]
    (compare num_a num_b)))

(defn compare-item
  [attrs a b]
  (log/info attrs)
  (let [sort (first attrs)
        attribute (first sort)
        order (second sort)
        value_a (get-in a attribute)
        value_b (get-in b attribute)]
    (log/info attribute)
    (log/info order)
    (log/info value_a)
    (log/info value_b)
    (cond
      (empty? attrs) 0
      (= value_a value_b) (compare-item
                           (rest attrs) a b)
      (= order :asc) (compare value_a value_b)
      (= order :desc) (- (compare value_a value_b))
      (= order :desc-number) (- (compare-as-number value_a value_b))
      (= order :asc-number) (compare-as-number value_a value_b))))

(defn sort-fn
  [q items]
  (sort
   (fn [a b]
     (let [attrs (mapv
                  (fn [[k v]]
                    [(to-keywords k) (keyword v)])
                  (partition 2 (:sort q)))]
       (compare-item attrs a b)))

   items))

(defn fix-keys
  [val]
  (-> val
      (util/to-json)
      (util/to-edn)))

(defn filter-aggregate
  [query aggregate]
  (if (seq query)
    (let [res (clojure-data/diff aggregate query)]
      (and (= (second res) nil)
           (= (nth res 2) query)))
    aggregate))

(defn impl->save-aggregate
  [ctx store aggregate]
  {:pre [aggregate]}
  (log/info "Emulated 'update-aggregate' dal function")
  (let [realm
        (lambda-ctx/realm ctx)

        service-name
        (lambda-ctx/get-service-name ctx)

        aggregate
        (fix-keys aggregate)]
    (swap! store
           #(update-in
             %
             [realm service-name :aggregate-store]
             (fn [v]
               (->> v
                    (filter
                     (fn [el]
                       (not= (:id el) (:id aggregate))))
                    (cons aggregate)
                    (sort-by (fn [{:keys [id]}] (str id)))))))))

(defn impl->get-snapshot
  [ctx store id]
  (util/d-time
   (format "Fetching snapshot aggregate %s" id)
   (let [realm
         (lambda-ctx/realm ctx)

         service-name
         (lambda-ctx/get-service-name ctx)

         aggregates
         (get-in @store
                 [realm service-name :aggregate-store])
         agg
         (->> aggregates
              (filter
               #(= (:id %) id))
              (first))]
     (if agg
       (log/info (format "Found snapshot with version: %s" (:version agg)))
       (log/info "Not snaphsot found"))
     agg)))

(defn impl->simple-search
  [ctx store query]
  {:pre [query]}
  (let [realm
        (lambda-ctx/realm ctx)

        service-name
        (lambda-ctx/get-service-name ctx)

        aggregates
        (get-in @store
                [realm service-name :aggregate-store])]
    (vec
     (filter
      #(filter-aggregate
        (dissoc query :query-id)
        %)
      aggregates))))

(defn impl->advanced-search
  [ctx store query]
  (let [realm
        (lambda-ctx/realm ctx)

        service-name
        (lambda-ctx/get-service-name ctx)

        aggregates
        (get-in @store
                [realm service-name :aggregate-store])

        apply-filter (if (:filter query)
                       (searchable-common/parse mock (:filter query))
                       (fn [_%] true))
        apply-search (if (:search query)
                       (partial search-fn query)
                       (fn [_%] true))
        apply-select (if (:select query)
                       (partial select-fn query)
                       (fn [%] %))
        apply-sort (if (:sort query)
                     (partial sort-fn query)
                     (fn [%] %))
        hits (->> aggregates
                  (filter apply-filter)
                  (filter apply-search)
                  (map apply-select)
                  (apply-sort)
                  (into []))
        to (+ (get query :from 0)
              (get query :size (count hits)))]
    {:total (count hits)
     :from  (get query :from 0)
     :size  (get query :size searchable-common/default-size)
     :hits  (subvec hits
                    (get query :from 0)
                    (if (> to (count hits))
                      (count hits)
                      to))}))

(def initial-state {:aggregate-store []})

(deftype MemorySearchableViewStore [config ^clojure.lang.Atom store]
  view-store/ViewStore
  (init [_this ctx]
    (log/info (str "init: " _this))
    (let [realm
          (lambda-ctx/realm ctx)

          service-name
          (lambda-ctx/get-service-name ctx)]
      (util/d-time
       (str "Intializing MemorySearchableViewStore with data"
            (count (:aggregate-store ctx [])))
       (reset! store
               (assoc-in
                {}
                [realm service-name]
                (merge
                 initial-state
                 (select-keys ctx [:aggregate-store]))))
       ctx)))
  (get-snapshot [_this ctx query]
    (impl->get-snapshot ctx store query))
  (save-aggregate [_this ctx aggregate]
    (log/info (str "save: " _this))
    (impl->save-aggregate ctx store aggregate))

  searchable-view-store/SearchableViewStore
  (advanced-search [_this ctx query]
    (impl->advanced-search ctx store query))
  (simple-search [_this ctx query]
    (log/info (str "search: " _this))
    (impl->simple-search ctx store query)))

(defn register
  [ctx & [config]]
  (view-store/register
   ctx
   {:implementation-class MemorySearchableViewStore
    :config (or config {})
    :args [(atom {})]
    :config-default {}
    :config-schema (m/schema
                    [:map])}))
