(ns lambda.core)

(defprotocol FilterChain
  (continue-filter [_this ctx request]))

(defprotocol LambdaFilter
  "Filter is used to filter out platform specific data from request. 
   They will be assigned to Runtime and runtime will be responsible 
   for initializing and running all filters.
   
   For example we will use filters to unwrap AWS lambda requests wrapper 
   so that we pass to handler only inmporatnt body of the request. 
   
   We can use also fiters to enable processing requests by uploading 
   data to S3 bucket. So we can create filter that wil download file 
   from S3 and use it as real request."
  (init-filter [_this ctx]
    "Filter intialization. This can be used for example to fetch  
     secrets from Vault. Usually best place to place things that 
     should be done once and are requed for filter.")
  (do-filter [_this ctx request ^FilterChain filter-chain]
    "Actual filtering of request. We should invoke filter-chain 
     as FilterChain accepting 2 parameters (ctx and request). FilterChain 
     will return response. We can use it for example for 
     AWS Api gataway request to assign propper status code and 
     wrapper required for it to work."))

(defprotocol LambdaHandler
  "Handler is responsible for actually handling request. This is where 
   we implement for example EDD logic. Reason for having multiple handlers 
   is that we can for example implement non EDD lambda function using AWS 
   custom runtime. We want to be able to use Filters and Runtime but have
   new Handler. 

   Handler implementation is recommented to accept filter in the constructor. 
   Handler. "
  (init-handler [this ctx]
    "Handler init method is responsible for instantiation and initializing all 
     filters by calling Filter/init function.")
  (do-handle [this ctx request]
    "Actial request handler implementation (i.e. EDD)"))

(defprotocol MiddlewareChain
  (continue-process [_this ctx request]))

(defprotocol LambdaMiddlware
  "This is middleware for handler. It will be used for 
   platform specific implementation related to handler. For example 
   in edd-core we will use it for implementing effect distribution. 
   We will want to have implementation for example that distributes 
   effects to SQS queues and we will want to have implementation 
   for distributing effects to Kafka or RabbitMQ. This will enable 
   us to run same EDD logic in different environments."
  (init-middleware [this ctx]
    "Middleware intialization. This can be used for example to fetch  
     secrets from Vault. Usually best place to place things that 
     should be done once and are requed for filter.")
  (do-process [this ctx request ^MiddlewareChain middleware-chain]
    "Actual midleware work for request. We should invoke filter-chain 
     as function accepting 2 parameters (ctx and request). Return of 
     middlware-chain function is resonse. We can use if for example 
     to distribute Effects for EDD handler implementaion."))

(defprotocol LambdaRuntime
  "It is responsible for barebone infrastructure implementation to enable 
   receiving re quests and sending responses. This can be simple as starting 
   Undertow or implementing AWS Custom runtime.

   Example: We will have AWS Custom Runtime that will receive requests. It will 
   parse request and invoke handler. Handler will have APIGW and SQS filters and 
   EddHandler. In case of API request will receive wrapper body of request that 
   we will strip and parse in APIGWFilter (SQS filter in this request will be skipped). 
   APIGWFilter will after request is procced wrap respons in map adding CORS Headers 
   and status field (As required by AWS APIGW specification). Runtime will after 
   processing return response back to AWS Lambda."
  (init-runtime [_this ctx]
    "It is responsible for invoking Handler/init function. And can be used 
     for fetching required items for handling requests (i.e. Fetching secrets). 
     This method should not be blocking and must return context.")
  (start-runtime [_this ctx]
    "Blocking method that is starting runtime. 
     In this function we would start for example undertow server or start 
     loop used for AWS Custom runtime. 
     Parameter ctx in this function will be output of init-runtime function.")
  (run-request [_this ctx request]
    "This function will actially process request. This function will be called 
     from within start-runtime or from ourside. 
     
     This function is responsible to bind lambda.request/*request* as atom. 
     This way all filter and handler can rely on request scoped atom."))

