(ns aws.filters.sqs
  (:require [lambda.util :as util]
            [clojure.tools.logging :as log]
            [clojure.string :as string]
            [lambda.jwt :as jwt]
            [lambda.core :as lambda-core]
            [aws.ctx :as aws-ctx])
  (:import (lambda.core LambdaFilter)))

(defn process-batch-item
  [ctx id item filter-chain]
  (try
    (let [body (util/to-edn (:body item))]
      (lambda-core/continue-filter filter-chain
                                   ctx
                                   body))
    (catch Exception e
      (log/error e (format "Unable to process message: %s" id))
      {:exception e})))

(defn aws-sqs-filter->do-filter
  "We are expecting batch of items here. Each item in 
   batch is completely independant request so we will threat it like that 
   and call continue filter for each item in batch."
  [ctx _config request filter-chain]
  (let [batch-items (:Records request)
        result (loop [items batch-items
                      failed-messages []]
                 (if (not (seq items))
                   failed-messages
                   (let [{:keys [messageId] :as item}
                         (first items)

                         {:keys [exception]}
                         (process-batch-item ctx messageId item filter-chain)

                         remaining
                         (rest items)]
                     (cond
                       (seq failed-messages)
                       (recur remaining
                              (conj failed-messages messageId))

                       exception
                       (recur remaining
                              (conj failed-messages messageId))

                       :else
                       (recur remaining
                              failed-messages)))))]
    (if (seq result)
      (throw (ex-info
              (format "Some messages where not processed: %s"
                      (string/join ", "
                                   result))
              {"batchItemFailures" (mapv
                                    (fn [r]
                                      {"itemIdentifier" r})
                                    result)}))
      {:success true})))

(defn aws-sqs-filter->filter-request
  [ctx config request filter-chain]
  (if (and
       (contains? request :Records)
       (= (:eventSource
           (first
            (:Records request)))
          "aws:sqs"))
    (aws-sqs-filter->do-filter ctx config request filter-chain)
    (do
      (log/info "Skipping AWSSqsFilter")
      (lambda-core/continue-filter filter-chain
                                   ctx
                                   request))))
(defn aws-sqs-filter->init
  [ctx]
  (-> ctx
      jwt/fetch-jwks-keys
      aws-ctx/init
      jwt/ctx->aws-user-pool))

(deftype AWSSqsFilter [config]
  LambdaFilter
  (init-filter [_this ctx]
    (aws-sqs-filter->init ctx))
  (do-filter [_this ctx request filter-chain]
    (aws-sqs-filter->filter-request ctx config request filter-chain)))
