(ns com.analytica.client
  (:require  [clj-http.client :as client]
             [cheshire.core :as json]))

(defmacro log-expr [e]
  `(let [e# ~e]
     (println (str "Expression " '~e " evaluates to " e# " of type " (.getClass e#) "\n|\n|"))
     e#))
;; ## Helpers to deal with handling responses and forming URLs


(def analytica-defaults {:host "localhost" :port 8123})

(defn- analytica-url []
  (str "http://" (:host analytica-defaults) ":" (:port analytica-defaults)))

(defn- url-join [& strings]
  (reduce  #(str % "/" %2) "" strings))

(defn- ->json [str]
  (json/parse-string str true))

(defn- resp-body [resp]
  (:body resp))

(defn resp-body->json [resp]
  (->json (resp-body resp)))

(defn resp-success? [resp]
  (= (:status resp) 200))

(defn query-success? [parsed-resp-body]
  (:success parsed-resp-body))

(defn read-body-if-success [resp]
  (when (resp-success? resp)
    (resp-body->json resp)))

(defn parse-results-if-success
  "After having processed a response body, determine if the action was a success, and, if so, read out the results"
  [parsed-resp-body]
  ;; first? so many nested lists...
  (when (query-success? parsed-resp-body)
    (first (map ->json (:result parsed-resp-body)))))
  
  

;; ## Step 0) Test

(defn ping
  "/ping"
  []
  (client/get (str (analytica-url) "/ping")))

(defn connection-status
  "/connectionStatus"
  []
  (client/get (str (analytica-url) "/connectionStatus")))



;; ## Step 1) Connect

(defn connect!
  "/connect/hostName/databaseName/databaseType/databasePort/ databaseUsername/databasePassword"
  ([db-name] (connect! db-name "localhost"))
  ([db-name host] (connect! db-name host "27017"))
  ([db-name host port] (connect! db-name host port "---" "+++" ))
  ([db-name host port username password]
     (connect! db-name host port username password "MongoDB"))
  ([db-name host port username password data-source-type]
     (client/get 
      (str (analytica-url) "/connect"
           (url-join host db-name data-source-type port username password)))))

(defn autoconnect!
  "/autoconnect/host/username/password"
  ([] (autoconnect! "localhost"))
  ([host] (autoconnect! host "---" "+++"))
  ([host username password]  
     (client/get
      (str (analytica-url) "/autoconnect"
           (url-join host username password)))))


;; ## Step 2) Go to town!

(defn get-coll  
  "/get/[colletion-name]"
  [collection-name]
  (client/get (str (analytica-url) "/get/" collection-name)))


(defn executeXL [xl-str]
  "/executeXL/[XL-expr]"
  (client/get (str (analytica-url) "/executeXL/" xl-str)))

;; ## Step 3) Clean Up

(defn disconnect!
  "/disconnect/hostName/databaseName/databaseType/databasePort/"
  ([db-name] (disconnect! db-name "localhost"))
  ([db-name host] (disconnect! db-name host "27017"))
  ([db-name host port] (disconnect! db-name host port "MongoDB"))
  ([db-name host port data-source-type]
     (client/get 
      (str (analytica-url) "/disconnect"
           (url-join host db-name data-source-type port)))))

(defn disconnect-all
  "/connectionStatus"
  []
  (client/get (str (analytica-url) "/disconnectall")))

;; # Macros

(defmacro with-connection
  "Opens a connection to a particular host and database, performs body,
   and closes the connection, returning the result of the body"
  [[db-name host port username password] & body]
  `(do
     (connect! ~db-name ~host ~port ~username ~password "MongoDB")
     (let [result# (do ~@body)]
       (disconnect! ~db-name ~host ~port "MongoDB")
       result#)))

(defmacro reduce-over-host-databases
  "Connects to all databases at a given host, performs a reduce over each
   database, then disconnects from each database, returning the result of
   the reduce"
  [[host username password] konj knil]
  `(let [konj# ~konj
         knil# ~knil]
     ;; ~port ;; why no port?
     (autoconnect! ~host ~username ~password)
     (let [db-names#
           (map #(:databaseName %)
                (:getdatabases
                 (parse-results-if-success
                  (read-body-if-success (executeXL "getDatabases()")))))
           result# (reduce konj# knil# db-names#)]
       (doseq [db-name# db-names#] (disconnect! db-name# ~host))
       result#)))
       


(defmacro parse-collection-query-results
  "Given a thunk that queries for a collection, parse the query results
   if the query is successful"
  [collection-query-thunk]
  `(let [coll-resp# (~collection-query-thunk)]
     (if (resp-success? coll-resp#)
       (parse-results-if-success (resp-body->json coll-resp#))
       (print (str "bad response :" coll-resp#)))))


  
