(ns cljs-time.tz.parser
  (:require
   [clojure.java.io :as io]
   [clojure.string :as string]
   ))

(defn parse-tz-line [[off rules fmt year month day time]]
  (let [[hh mm ss] (string/split off #":")
        sign (if (.startsWith hh "-") :- :+)
        hh (string/replace hh #"^\+|-" "")]
    [(vec (concat [sign] (map #(if % (Integer. %) 0) [hh mm ss])))
     rules
     fmt
     (when year (Integer. year))
     month
     day
     time]))

(defn parse-tz-info [tz-name]
  (with-open [r (io/reader (io/resource (str "tz/" tz-name)))]
    (doall
     (->> (line-seq r)
          (remove #(or (empty? %) (.startsWith % "#")))
          (map #(string/replace % #"#.*$" ""))
          (map (fn [s]
                 (let [level (count (first (re-seq #"^\t+" s)))
                       [head :as data] (string/split s #"[\t\s]+")
                       what? (cond (= "Rule" head) :rule
                                   (= "Zone" head) :zone
                                   (= "Link" head) :link
                                   (= "" head) :zone-info)]
                   (when-not (every? nil? data)
                     {:type what?
                      :data
                      (condp = what?
                        :rule (let [[_ name when? & more] data]
                                [name [when? (vec more)]])
                        :zone (let [[_ name & more] data]
                                [name [(parse-tz-line more)]])
                        :link (rest data)
                        (parse-tz-line (rest data)))
                      :level level}))))
          (remove nil?)
          (reduce (fn [init {:keys [type data] :as x}]
                    (let [[name val] data]
                      (condp = type
                        :rule (update-in init [:rules name] (fnil conj []) val)
                        :zone (update-in init [:zones] conj [name val])
                        :link (assoc-in init [:links val] name)
                        (update-in init [:zones]
                                   #(vec (concat (butlast %)
                                                 [(update-in (last %) [1]
                                                             conj data)]))))))
                  {:rules {} :zones [] :links {}})
          (#(update-in % [:zones] (partial into {})))))))

(defn str-def [name m]
  (str "(def " name "\n"
       (with-out-str (clojure.pprint/pprint m))
       ")\n\n"))

(defn gen-tz-data [f]
  (let [tz (->> ["africa" "antarctica" "asia" "australasia" "etcetera" "europe"
                 "northamerica" "pacificnew" "southamerica"]
                (map parse-tz-info)
                (reduce (fn [init {:keys [rules zones links]}]
                          (-> init
                              (update-in [:rules] merge rules)
                              (update-in [:zones] merge zones)
                              (update-in [:links] merge links)))
                        {:rules {} :zones {} :links {}}))]
    (spit f ";;; This file is generated by cljs-time.tz.parser
;;; do not edit by hand.\n\n(ns cljs-time.tz.data)\n")
    (spit f (str-def "rules" (:rules tz)) :append true)
    (spit f (str-def "zones" (:zones tz)) :append true)
    (spit f (str-def "links" (:links tz)) :append true)))


(defn -main [& args]
  (println "Rebuilding tz data")
  (gen-tz-data "src/cljs_time/tz/data.cljx"))
