(ns atomist.deps
  (:require [cljs-node-io.core :as io]
            [cljs-node-io.fs :as fs]
            [atomist.cljs-log :as log]
            [atomist.sdmprojectmodel :as sdm]
            [atomist.sha :as sha]
            [atomist.json :as json]
            [atomist.api :as api]
            [atomist.graphql :as graphql]
            [goog.string :as gstring]
            [goog.string.format]
            [cljs.core.async :refer [<! timeout] :as async])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn- get-param [x s]
  (->> x (filter #(= s (:name %))) first :value))

(defn- off-target? [fingerprint target]
  (and (= (:name fingerprint) (:name target))
       (not (= (:sha fingerprint) (:sha target)))))

(defn- policy-order
  "when there are overlaps, use manual config overrides latest semver, which over-rides latest semver used"
  [v]
  (letfn [(is [x] (complement (fn [y] (= x y))))]
    (count (take-while (is v) ["latestSemVerUsed" "latestSemVerAvailable" "manualConfiguration"]))))

(defn policy-type [configuration]
  (->> configuration :parameters (filter #(= (:name %) "policy")) first :value))

(defn dependencies [configuration]
  (->> configuration :parameters (filter #(= (:name %) "dependencies")) first :value))

(defn all-strings?
  [d]
  (let [data (try (cljs.reader/read-string d)
                  (catch :default ex (log/warnf "invalid:  %s" d) false))]
    (and data (coll? data) (every? string? data))))

(defn deps-array?
  [d]
  (let [data (try (cljs.reader/read-string d) (catch :default ex (log/warnf "invalid:  %s" d)))]
    (and data (coll? data) (every? #(and (coll? %) (= 2 (count %)) (-> % second string?)) data))))

(defn validate-policy [configuration]
  (case (policy-type configuration)
    "latestSemVerUsed" (if (all-strings? (dependencies configuration))
                         configuration
                         (assoc configuration :error "latestSemVerUsed dependencies must be an array of Strings"))
    "latestSemVerAvailable" (if (all-strings? (dependencies configuration))
                              configuration
                              (assoc configuration :error "latestSemVerAvailable dependencies must be an array of Strings"))
    "manualConfiguration" (if (deps-array? (dependencies configuration))
                            configuration
                            (assoc configuration :error "manualConfiguration dependencies configuration invalid"))))

(defn set-up-target-configuration
  "middleware to construct a manualConfiguration
     from a dependency configuration containing a '[lib version]' string"
  [handler]
  (fn [request]
    (log/infof "set up target dependency to converge on [%s]" (:dependency request))
    (handler (assoc request
               :configurations [{:parameters [{:name "policy"
                                               :value "manualConfiguration"}
                                              {:name "dependencies"
                                               :value (gstring/format "[%s]" (:dependency request))}]}]))))

(defn mw-validate-policy
  "middleware to validate an edn deps policy
    all configurations with a policy=manualConfiguration should have a dependency which is an application/json map
    all configurations with other policies use a dependency which is an array of strings"
  [handler]
  (fn [request]
    (try
      (let [configurations (->> (:configurations request)
                                (map validate-policy))]
        (if (->> configurations
                 (filter :error)
                 (empty?))
          (handler request)
          (api/finish request :failure (->> configurations
                                            (map :error)
                                            (interpose ",")
                                            (apply str)))))
      (catch :default ex
        (log/error ex)
        (api/finish request :failure (-> (ex-data ex) :message))))))

(defn configs->policy-map
  "  policy-type is one of latestSemVerUsed, latestSemVerAvailable, or manualConfiguration
     dependencies is an edn string which must parse to either [lib1,lib2,...] or [[lib \"version\"],...]

     returns {library {:policy policy-type :version v}} where version is optional (might be added in later)"
  [configurations]
  (letfn [(read-dependency-string [c policy] (->> (cljs.reader/read-string (dependencies c))
                                                  (map str)
                                                  (reduce #(assoc %1 %2 {:policy policy}) {})))
          (read-manual-dependencies [c] (->> (cljs.reader/read-string (dependencies c))
                                             (reduce (fn [agg [k v]] (assoc agg (str k) {:policy :manualConfiguration
                                                                                         :version v})) {})))]
    (->> configurations
         (filter (constantly true))                         ;; TODO only apply policies applicable to current Repo filters
         (sort-by (comp policy-order policy-type))
         (map (fn [configuration] (case (policy-type configuration)
                                    "latestSemVerUsed" (read-dependency-string configuration :latestSemVerUsed)
                                    "latestSemVerAvailable" (read-dependency-string configuration :latestSemVerAvailable)
                                    "manualConfiguration" (read-manual-dependencies configuration))))
         (apply merge))))

(defn get-latest [o type dependency]
  (go
   (try
     (-> (<! (api/graphql->channel o graphql/query-latest-semver {:type type
                                                                  :name dependency}))
         :body
         :data
         :fingerprintAggregates)
     (catch :default ex
       (log/error "semver query " ex)))))

(defn version-channel
  "channel to a plan element
    dictating the correct next version for a target dependency

    this assumes that dependency fingerprints use json encoded [library version] to represent their data
      - this is true for leiningen, and npm.  TODO However, maven fingerprints are not computed this way.

    policy-map is {library {:policy policy-type :version v}}
    fp is {:keys [name type data]}

    returns a plan element {:fingerprint current-fingerprint :target target-fingerprint}"
  [o policy-map fp]
  (letfn [(transform-data [m] (assoc m :data (json/->obj (:data m))))]
    (go
     (let [dependency (gstring/replaceAll (:name fp) "::" "/")
           {:keys [policy] :as d} (policy-map dependency)

           plan-element {:target (case policy
                                   :latestSemVerAvailable (-> (<! (get-latest o (:type fp) (:name fp)))
                                                              :latestSemVerAvailable
                                                              (transform-data))
                                   :latestSemVerUsed (-> (<! (get-latest o (:type fp) (:name fp)))
                                                         :latestSemVerUsed
                                                         :fingerprint
                                                         (transform-data))
                                   :manualConfiguration (let [data [dependency (:version d)]]
                                                          {:name (gstring/replaceAll dependency "/" "::")
                                                           :sha (sha/sha-256 (json/->str data))
                                                           :data data})
                                   {})
                         :fingerprint fp}]
       plan-element))))

(defn apply-policy-targets
  "returns a channel which will emit a :complete value after applying all version policies to a Repo
     Steps are roughly
       - use configuration to construct the Policy to apply
       - emit plan elements from a version-channel (which may fetch live latest data from the graph)
       - check whether any current fingerprint are off target
       - apply plan elements that represent off-target deps inside of a PR editor"
  [{:keys [project configurations fingerprints] :as request}
   branch-name
   apply-name-version-library-editor]
  (let [policy-map (configs->policy-map configurations)]
    (go
     (let [plan (<! (->> (for [{current-data :data :as fingerprint} fingerprints]
                           (version-channel request policy-map fingerprint))
                         (async/merge)
                         (async/reduce (fn [plan {:keys [fingerprint target] :as off-target}]
                                         (if (and target (off-target? fingerprint target))
                                           (conj plan off-target)
                                           plan)) [])))]
       (log/info "plan " plan)
       (let [body (->> plan
                       (map (fn [{{current-data :data} :fingerprint
                                  {target-data :data} :target :as p}]
                              (gstring/format "off-target %s %s/%s -> %s/%s"
                                              (-> p :fingerprint :name)
                                              (nth current-data 0) (nth current-data 1)
                                              (nth target-data 0) (nth target-data 1))))
                       (interpose "\n")
                       (apply str))
             pr-opts {:branch branch-name
                      :target-branch (-> request :ref :branch)
                      :title (gstring/format "%s skill requesting dependency change" (-> fingerprints first :type))
                      :body body}]
         (<! ((sdm/edit-inside-PR
               (fn [p]
                 (go
                  (doseq [{{:keys [data]} :target} plan]
                    (log/info "applying " data " : " (<! (apply-name-version-library-editor p (nth data 0) (nth data 1)))))
                  :done))
               pr-opts) project))))
     :complete)))

(defn- handle-push-event [request compute-fingerprints mw-validate-policy]
  ((-> (api/finished :message "handling Push" :success "successfully handled Push event")
       (api/send-fingerprints)
       (api/run-sdm-project-callback compute-fingerprints)
       (mw-validate-policy)
       (api/extract-github-token)
       (api/create-ref-from-push-event)) request))

(defn fp-command-handler [request just-fingerprints]
  ((-> (api/finished :message "handling extraction CommandHandler")
       (api/show-results-in-slack :result-type "fingerprints")
       (api/run-sdm-project-callback just-fingerprints)
       (api/create-ref-from-first-linked-repo)
       (api/user-should-choose-one-linked-repo)
       (api/extract-linked-repos)
       (api/extract-github-user-token)
       (api/set-message-id)) (assoc request :branch "master")))

(defn update-command-handler [request compute-fingerprints validate-parameters]
  ((-> (api/finished :message "handling application CommandHandler")
       (api/show-results-in-slack :result-type "fingerprints")
       (api/run-sdm-project-callback compute-fingerprints)
       (api/create-ref-from-first-linked-repo)
       (api/user-should-choose-one-linked-repo)
       (api/extract-cli-parameters [[nil "--slug SLUG" "org/repo"]])
       (api/extract-linked-repos)
       (api/extract-github-user-token)
       (validate-parameters)
       (api/set-message-id)) (assoc request :branch "master")))

(defn deps-handler [data sendreponse [show just-fingerprints] [update compute-fingerprints validate-parameters] mw-validate-policy]
  (api/make-request
   data
   sendreponse
   (fn [request]
     (cond
       ;; handle Push events
       (contains? (:data request) :Push)
       (handle-push-event request compute-fingerprints mw-validate-policy)

       (= show (:command request))
       (fp-command-handler request just-fingerprints)

       (= update (:command request))
       (update-command-handler request compute-fingerprints validate-parameters)))))
