(ns atomist.sdmprojectmodel
  (:require ["@atomist/automation-client" :as ac]
            ["@atomist/automation-client/lib/operations/support/editorUtils" :as editor-utils]
            [cljs.core.async :refer [<! timeout] :as async]
            [atomist.promise :as promise]
            [atomist.cljs-log :as log]
            [goog.string :as gstring]
            [goog.string.format]
            [cljs-node-io.proc :as proc]
            [cljs-node-io.core :refer [slurp]]
            [clojure.string :as s])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(def x (. ac -PlainLogging))
(set! (.. x -console -level) "debug")

(defn enable-sdm-debug-logging []
  ((. ac -configureLogging) x))

(defn update-project!
  "asynchronously update sdm Project.id.sha (uses `git rev-list -1 HEAD --`)
    also runs `git config user.name atomist.bot` and `git config user.email bot@atomist.com` to make sure cloned project is ready for pushes
    returns chan that will emit the updated Project"
  [p]
  (go
   (if (= "HEAD" (.. ^js p -id -sha))
     (set! (.-sha ^js (.-id ^js p)) (:sha (<! (promise/from-promise ((.gitStatus ^js p)))))))
   (<! (promise/from-promise (.setUserConfig ^js p "atomist-bot" "bot@atomist.com")))))

;; Note:  this is important when running gcf functions because this can leak memory when the /tmp filesystem is in memory
(defn recursive-delete
  "spawn rm to clean up Project.baseDir"
  [p]
  (go
   (let [basedir (. ^js p -baseDir)]
     (if basedir
       (let [[error stdout stderr] (<! (proc/aexec (gstring/format "rm -fr %s" basedir)))]
         (if error
           (log/error error stderr)))))))

(defn do-with-shallow-cloned-project
  "project middleware that does a shallow clone of the ref in a tmp directory (uses automation client DirectoryManager)

    returns chan that will emit the value returned by the async project-callback"
  [project-callback token ref]
  (go
   (let [p (<! (promise/from-promise
                (.cloned (.-GitCommandGitProject ac)
                         #js {:token token}
                         (.from (.-GitHubRepoRef ac) (clj->js ref))
                         #js {:alwaysDeep false})))]
     (if (. ^js p -baseDir)
       (do
         (log/debugf "tmpdir %s" (. ^js p -baseDir))
         (<! (update-project! p))
         (let [result (<! (project-callback p))]
           (<! (recursive-delete p))
           result))
       (log/warn "project clone failure")))))

(defn commit-then-push
  "project middleware for wrapping an project editor in a commit then push flow
     project-callback returns chan that will emit either :done or :failure"
  [project-callback commit-message]
  (fn [p]
    (go
     (<! (project-callback ^js p))
     (<! (promise/from-promise (.commit ^js p commit-message)))
     (<! (promise/from-promise (.push ^js p))))))

(defn remote?->chan
  "asynchronously look for a remote branch ref on origin

    channel emits nil or the short name of the remote branch ref"
  [project branch]
  (go
   (let [[err stdout stderr] (<! (proc/aexec "git ls-remote --heads" {:cwd (. ^js project -baseDir)}))]
     (not (empty?
           (->> (s/split-lines (if (string? stdout) stdout (slurp stdout)))
                (map #(second (re-find #".*refs/heads/(.*)$" %)))
                (filter #(= branch %))))))))

(defn checkout-branch->chan
  "fetch a branch with depth 1, update the ref head, and then move the working copy to point at this branch
    this is used to move working copy to a different branch from the one that was cloned
    this version not rely on any remote tracking branches, and can only push - not set up to pull, or merge"
  [project branch]
  (let [opts {:cwd (. ^js project -baseDir)}]
    (letfn [(handle-errors [[error _ stderr]]
              (if error
                (do (log/error error stderr)
                    :failed)
                :done))]
      ;; get fetch will fail with `Refusing to fetch into current branch refs/heads/master of non-bare repository` if
      ;; the currently checked out branch is equal to branch
      (go
       (if (= :done (handle-errors (<! (proc/aexec (gstring/format "git fetch origin '%s:%s' --depth 1" branch branch) opts))))
         (handle-errors (<! (proc/aexec (gstring/format "git checkout '%s'" branch) opts)))
         :done)))))

(defn- hasBranch?
  "the sdm hasBranch can only check for local branches.  This function also checks for remote refs.
    returns channel that emits boolean (true if there is a local or remote origin branch with this name)"
  [project branch-name]
  (go
   (if (<! (promise/from-promise
            (.hasBranch ^js project branch-name)
            identity
            (fn [error] (throw (ex-info "sdm failure checking branch" {:failure error})))))
     true
     (<! (remote?->chan project branch-name)))))

(defn edit-inside-PR
  "generate a function to execute a project callback wrapped by an SDM PullRequest
    project-callback is (p) => channel emitting :done if successful

    - outer project callback returns a chan that emits one of :done :failure :raised :skipped

    - most of these Project api calls return the this if successful (Project can be mutated)
    - since these api calls wrap git cli commands, the Promises can fail if git cli command
      returns non-zero at any point"
  [project-callback {:keys [branch target-branch body title]}]
  (fn [p]
    (letfn [(handle-sdm-failure [{:keys [done failure]}]
              (if failure
                (throw (ex-info "sdm project operation failure" {:failure failure}))
                {:result done}))
            (handle-sdm-commit-failure [{:keys [done failure]}]
              (if failure
                (throw (ex-info "empty commits do not require a Push" {:skip "empty commit"}))
                {:result done}))
            (promise->channel [promise] (promise/from-promise
                                         promise
                                         (fn [v] {:done v})
                                         (fn [error] {:failure error})))]
      (let [cloned-branch (.. ^js p -id -branch)]
        (log/debugf "cloned-branch: %s, target-branch: %s, edit-branch: %s" cloned-branch target-branch branch)
        ;; case 1:  new push to base, no existing branch/PR -> create new edit-branch and make change
        ;; case 2:  new push to base, branch exists -> rebase edit-branch and edit again?
        ;; case 3:  new push to edit-branch, if it's the rebase skill, we should run again.
        ;;          It's probably safe to run again in all cases because it'll be an empty commit so nothing would get Pushed.
        (go
         (try
           ;; STEP 1 - set up working copy
           (if (not (= cloned-branch branch))
             (let [response (<! (hasBranch? p branch))]
               (if (not response)
                 (handle-sdm-failure (<! (promise->channel (.createBranch ^js p branch)))) ;; create and checkout
                 (if (not (= :done (<! (checkout-branch->chan p branch))))
                   (throw (ex-info "unable to checkout" {:branch branch
                                                         :cloned-branch cloned-branch
                                                         :message ""}))))))
           ;; by now project should represent a local git copy with HEAD ref pointed at branch
           ;; STEP 2 - run project callback in working copy
           ;;   only if the project-callback returns :done will we commit, push, and raise pull request
           ;;   commit will not create empty commits; instead, it raise :skip if it sees no changes
           ;;   raising a PullRequest when one is already open between these branches, is not a failure
           (let [results (<! (project-callback p))]
             (if (= :done results)
               (do
                 (handle-sdm-commit-failure (<! (promise->channel (.commit ^js p body))))
                 (handle-sdm-failure (<! (promise->channel (.push ^js p))))
                 ;; idempotent - the SDM does not fail if this PR already exists
                 (handle-sdm-failure (<! (promise->channel (.raisePullRequest ^js p title body target-branch))))
                 :raised)
               (throw (ex-info "SDM project-callback did not finish normally" {:project p :done results}))))
           (catch :default ex
             (if (:skip (ex-data ex))
               (do
                 (log/debug (:skip (ex-data ex)))
                 :skipped)
               (do
                 (log/error ex (ex-data ex))
                 :failure)))))))))

(defn do-with-files
  "middleware for project handling - wrap project-callback in file iterator
     project-callback returns chan with an untyped value"
  [file-callback & patterns]
  (fn [p]
    (go
     (let [data (atom [])
           callbacks-over-all-patterns
           (<! (async/reduce
                conj []
                (async/merge
                 (for [pattern patterns]
                   (promise/from-promise
                    (.doWithFiles
                     (.-projectUtils ac)
                     p
                     pattern
                     (fn [f]
                       (log/debug (goog.string/format "Processing file: %s" (.-path f)))
                       (promise/chan->promise (go
                                               (let [d (<! (file-callback f))]
                                                 (swap! data conj d)
                                                 d))))))))))]
       (if @data @data :done)))))

(defn get-content
  " returns chan with file content String or {:failure error}"
  [f]
  (promise/from-promise (.getContent ^js f)))

(defn set-content
  " returns chan with the updated File #js object"
  [f content]
  (promise/from-promise (.setContent ^js f content)))
