(ns atomist.deps
  (:require [cljs-node-io.core :as io]
            [cljs-node-io.fs :as fs]
            [atomist.cljs-log :as log]
            [atomist.sdmprojectmodel :as sdm]
            [cljs.core.async :refer [<! timeout]]
            [atomist.sha :as sha]
            [atomist.json :as json]
            [goog.string :as gstring]
            [goog.string.format])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn- get-param [x s]
  (->> x (filter #(= s (:name %))) first :value))

(defn- off-target? [fingerprint target]
  (and (= (:name fingerprint) (:name target))
       (not (= (:sha fingerprint) (:sha target)))))

(defn configs->target-map
  "take configs with name/version parameters and convert them to an FP structure like:

   {:name \"metosin::compojure-api\",\n  :version \"1.1.12\"
    :sha \"abbd49a09f20626caf3a43eba07899c9502d7e5323b3beffa6995389272efe32\",
    :data [\"metosin/compojure-api\" \"1.1.12\"],
    :library \"metosin/compojure-api\"}

    configs that do not contain name/version parameters are filtered out"
  [configs]
  (->> configs
       (filter #(and (not (empty? (:parameters %)))))
       (map :parameters)
       (map (fn [x] (assoc {} :name (get-param x "name") :version (get-param x "version"))))
       (filter #(and (:name %) (:version %)))
       (map #(let [data [(:name %) (:version %)]] (assoc % :sha (sha/sha-256 (json/->str data)) :data data)))
       (map #(assoc % :name (gstring/replaceAll (:name %) "/" "::")
                      :library (:name %)))
       (into [])))

(defn apply-name-version-fingerprint-target
  "iterate over configuration targets and fingerprints and check for off-target fingerprints
    we wrap any edits in sdm/commit-then-PR so this function might create several PRs,
    depending on how many off target leiningen versions are present.  "
  [{:keys [project configurations fingerprints] :as request}
   apply-name-version-library-editor]
  (log/info "configurations " configurations)
  (go
   (let [targets (configs->target-map configurations)]
     (doseq [{current-data :data :as fingerprint} fingerprints]
       (doseq [{target-data :data :as target} targets]
         (when (off-target? fingerprint target)
           (let [body (gstring/format "off-target %s %s/%s -> %s/%s"
                                      (:displayType fingerprint)
                                      (nth current-data 0) (nth current-data 1)
                                      (nth target-data 0) (nth target-data 1))]
             (log/info body)
             (<! (apply-name-version-library-editor
                  project
                  {:branch (:library target)
                   :target-branch (-> request :ref :branch)
                   :title (gstring/format "%s: %s skill requesting change" (:library target) (:displayType fingerprint))
                   :body body}
                  (nth target-data 0)
                  (nth target-data 1)))))))
     :complete)))

