(ns atomist.sdmprojectmodel
  (:require ["@atomist/sdm" :as sdm]
            ["@atomist/automation-client" :as ac]
            ["@atomist/automation-client/lib/operations/support/editorUtils" :as editor-utils]
            [cljs.core.async :refer [<! timeout]]
            [atomist.promise :as promise]
            [atomist.cljs-log :as log]
            [goog.string :as gstring]
            [goog.string.format])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(def x (. ac -PlainLogging))
(set! (.. x -console -level) "debug")

(defn enable-sdm-debug-logging []
  ((. ac -configureLogging) x))

(defn do-with-shallow-cloned-project
  "does a shallow clone of the ref in some tmp directory

    returns chan that will emit the value returned by the async project-callback"
  [project-callback token ref]
  (promise/from-promise
   (.doWithProject
    (.-CloningProjectLoader sdm)
    (clj->js {:credentials {:token token}
              :id (.from (.-GitHubRepoRef ac) (clj->js ref))
              :readOnly false
              :cloneOptions {:alwaysDeep false}})
    (fn [p]
      (log/info "shallow cloned project at " (. ^js p -baseDir))
      (promise/chan->promise
       (go
        (<! (promise/from-promise (.setUserConfig ^js p "atomist-bot" "bot@atomist.com")))
        (<! (project-callback ^js p))))))))

(defn commit-then-push
  "  middleware for project handling - wrap project-callback in commit, push
     project-callback returns chan that will emit either :done or :failure"
  [project-callback commit-message]
  (fn [p]
    (log/info "commit then push " p)
    (go
     (<! (project-callback ^js p))
     ;; TODO
     #_(<! (promise/from-promise (.call (.-createAndPushBranch editor-utils) p (clj->js {:branch "master"}))))
     (<! (promise/from-promise (.commit ^js p commit-message)))
     (<! (promise/from-promise (.push ^js p))))))

(defn commit-then-PR
  "  middleware for project handling - wrap project-callback in commit, push
     project-callback returns chan that will emit either :done or :failure
     params
       opts - {:keys [branch targetBranch body title commit-message]}"
  [project-callback {:keys [branch target-branch body title]}]
  (fn [p]
    (log/infof "commit then PR %s -> %s" target-branch branch)
    (go
     (<! (project-callback p))
     ;; TODO
     #_(<! (promise/from-promise (.call (.-raisePR editor-utils) p (clj->js (merge
                                                                             opts
                                                                             {:targetBranch "master"
                                                                              :autoMerge {:mode "[auto-merge-method:merge]"
                                                                                          :method "[auto-merge:on-approve]"}})))))
     (let [body (gstring/format "%s\n[auto-merge-method:merge] [auto-merge:on-approve]" body)]
       (log/info "create branch " branch)
       (<! (promise/from-promise (.createBranch ^js p branch)))
       (let [commit-result (<! (promise/from-promise (.commit ^js p body)))]
         (if (not (contains? commit-result :failure))
           (do
             (<! (promise/from-promise (.push ^js p)))
             (log/infof "create PR titled %s for branch %s -> %s" title branch target-branch)
             (<! (promise/from-promise (.raisePullRequest ^js p title body target-branch)))
             {:message "created PR"})
           {:message "nothing to commit"}))))))

(defn do-with-files
  "  middleware for project handling - wrap project-callback in file iterator
     project-callback returns chan with an untyped value"
  [file-callback pattern]
  (fn [p]
    (promise/from-promise
     (.doWithFiles
      (.-projectUtils ac)
      p
      pattern
      (fn [f]
        (log/debug (goog.string/format "Processing file: %s" (.-path f)))
        (promise/chan->promise (go (<! (file-callback f)))))))))

(defn get-content
  " returns chan with file content String or {:failure error}"
  [f]
  (promise/from-promise (.getContent ^js f)))

(defn set-content
  " returns chan with the updated File #js object"
  [f content]
  (promise/from-promise (.setContent ^js f content)))
