(ns atomist.container
  (:require [atomist.cljs-log :as log]
            [atomist.promise :as promise]
            [atomist.api :as api]
            [cljs-node-io.core :as io]
            [atomist.json :as json]
            [cljs.core.async :refer [<! >! chan]]
            [atomist.meta :as meta]
            [goog.object :as g]
            [atomist.topics])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn read-atomist-payload [handler]
  (letfn [(payload->owner [{:keys [data]}]
            (or (-> data :Push first :repo :owner)
                (-> data :Tag first :commit :repo :owner)))
          (payload->repo [{:keys [data]}]
            (or (-> data :Push first :repo :name)
                (-> data :Tag first :commit :repo :name)))]
    (fn [request]
      (go
        (api/trace "read-atomist-payload")
        (try
          (let [payload (-> (io/file (:payload request))
                            (io/slurp)
                            (json/->obj)
                            (api/event->request))]
            (log/info "extensions " (:extensions payload))
            (log/info "secrets " (map :uri (:secrets payload)))
            (log/info "data " (:data payload))
            (if (contains? (:data payload) :Push)
              (<! (handler
                   (-> request
                       (assoc :owner (payload->owner payload) :repo (payload->repo payload))
                       (merge payload {:api-key (->> payload :secrets (filter #(= "atomist://api-key" (:uri %))) first :value)}))))
              (do
                (when-let [f (io/file (:dir request))]
                  (log/infof "%s - %s" (.getPath f) (.exists f))
                  (log/infof "Tag %s - %s" (-> request :data :Tag :name) (-> request :data :Tag :description)))
                (<! (api/finish request :success "not building new Tag event")))))
          (catch :default ex
            (log/error ex)
            request))))))

(defn create-logger [handler]
  (fn [request]
    (go
      (try
        (api/trace "create-logger")
        (if (.. js/process -env -TOPIC)
          (log/create-logger (:correlation-id request) "unknown" (-> request :team-id)))
        (log/infof "----> starting %s %s %s (%s)" meta/module-name meta/version meta/generated-at meta/tag)
        (<! (handler request))
        (catch :default ex
          (log/error ex)
          request)))))

(defn check-environment [handler]
  (letfn [(env [& envs]
            (reduce (fn [agg x]
                      (or agg (g/get (. js/process -env) x))) false envs))]
    (fn [request]
      (go
        (api/trace "check-environment")
        (try
          (let [request
                (cond-> request
                  (env "ATOMIST_WORKSPACE_ID" "WORKSPACE_ID") (assoc :team-id (env "ATOMIST_WORKSPACE_ID" "WORKSPACE_ID"))
                  (env "ATOMIST_GRAPHQL_ENDPOINT" "GRAPHQL_ENDPOINT") (assoc :graphql-endpoint (env "ATOMIST_GRAPHQL_ENDPOINT" "GRAPHQL_ENDPOINT"))
                  (env "ATOMIST_PAYLOAD" "PAYLOAD") (assoc :payload (env "ATOMIST_PAYLOAD"))
                  (env "ATOMIST_STORAGE" "STORAGE") (assoc :bucket-name (nth (re-find #"gs://(.*)" (env "ATOMIST_STORAGE" "STORAGE")) 1))
                  (env "ATOMIST_TOPIC" "TOPIC") (assoc :topic (env "ATOMIST_TOPIC" "TOPIC"))
                  (env "ATOMIST_CORRELATION_ID") (assoc :correlation-id (env "ATOMIST_CORRELATION_ID")))]
            (if (and (:team-id request) (:topic request) (:graphql-endpoint request) (:payload request) (:bucket-name request) (:correlation-id request))
              (do
                (log/info "environment:  " (select-keys request [:team-id :topic :graphql-endpoint :payload :bucket-name]))
                (<! (handler (assoc request :sendreponse (partial atomist.topics/sendreponse (:topic request))))))
              (do
                (log/warnf "environment is missing some of WORKSPACE_ID GRAPHQL_ENDPOINT ATOMIST_PAYLOAD ATOMIST_CORRELATION_ID or ATOMIST_STORAGE %s" (str request))
                (<! (api/finish request :failure "environment is missing some of WORKSPACE_ID GRAPHQL_ENDPOINT ATOMIST_PAYLOAD or ATOMIST_STORAGE")))))
          (catch :default ex
            (log/error ex)
            request))))))

(defn- exit [exit-code]
  (if (not (= "true" (.. js/process -env -IN_REPL)))
    (.exit js/process 0)))

(defn ^:api make-container-request
  [handler]
  (fn [request]
    (let [done-channel (chan)]
      ;; handler chain exists when this go block completes
      (go
        (try
          (<! (handler request))
          (>! done-channel "done")
          (catch :default ex
            ;; no guarantee that the response topic is active here (which means logging might not be working either)
            (log/error ex)
            (<! (api/failed-status request))
            (>! done-channel :failed))))
      (let [prom (promise/chan->promise done-channel)]
        (.info js/console prom)
        (.catch
         (.then
          prom
          (fn [result]
            (.info js/console result)
            (exit 0)))
         (fn [error]
           (.error js/console error)
           (exit 1)))))))

(def mw-make-container-request
  (api/compose-middleware
   [read-atomist-payload]
   [create-logger]
   [check-environment]
   [make-container-request]))