(ns atomist.git
  (:require [cljs.core.async :refer [<! >! timeout chan close!]]
            [goog.string :as gstring]
            [goog.string.format]
            [cljs-node-io.proc :as proc]
            [atomist.cljs-log :as log]
            [clojure.string :as s]
            ["os" :as os]
            ["tmp" :as tmp])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn from-git [context args]
  (go
    (log/debug "run " (->> args
                           (map #(-> %
                                     (s/replace #"https://(.*:x-oauth-basic)@(.*)" "$2")
                                     (s/replace #"https://(atomist:.*)@(.*)" "$2")))
                           (into [])))
    (let [ported-child-proc (proc/aexec (str "git " (apply str (interpose " " args))) {:cwd (:path context)})
          [err stdout stderr] (<! ported-child-proc)]
      (if err
        (merge context
               {:error err
                :exit-code (. err -code)
                :stderr stderr
                :stdout stdout})
        (merge context {:stdout stdout
                        :stderr stderr})))))

(defn clone-ref [{:keys [path branch token owner repo]}]
  ;; need success
  ["clone"
   (gstring/format "https://%s@github.com/%s/%s.git"
                   (if (s/starts-with? token "v1.")
                     (str "atomist:" token)
                     (str token ":x-oauth-basic"))
                   owner repo)
   path
   "--depth" "1"
   "--branch" (or branch "master")])

(defn sha [_]
  ;; need stdout
  ["rev-list" "-1" "HEAD" "--"])

(defn config [_ n v]
  ;; need success
  ["config" n v])

(defn add-all [_]
  ["add" "."])

(defn commit [_ message]
  ;; need success
  ["commit" "-m" (gstring/format "'%s'" message)])

(defn push [{:keys [branch]}]
  ["push" "--set-upstream" "origin" branch])

(defn has-branch [_ branch]
  ;; need branch to be in stdout
  ["branch" "--list" branch])

(defn create-branch [_ branch]
  ;; need both to be successful
  ["branch" branch])

(defn checkout [_ branch]
  ["checkout" branch "--"])

(defn ls-remote [_]
  ["ls-remote" "--heads"])

(defn fetch-branch [_ branch]
  ;; for refspecs, keep local and remote branches the same
  ["fetch" "origin" (gstring/format "'%s:%s'" branch branch) "--depth" "1"])

(defn checkout-branch [_ branch]
  ["checkout" (gstring/format "'%s'" branch)])

(defn checkout-force [_ branch]
  ["checkout" "--force" (gstring/format "'%s'" branch)])

(defn porcelain-status [_]
  ["status" "--porcelain"])