(ns atomist.graphql-channels
  (:require [atomist.json :as json]
            [atomist.graphql :as graphql]
            [cljs.core.async :refer [>! <! timeout chan]]
            [cljs.pprint :refer [pprint]]
            [http.client :as client]
            [atomist.promise :as promise]
            [atomist.cljs-log :as log]
            [com.rpl.specter :as specter]
            [atomist.time]
            [goog.string :as gstring]
            [goog.string.format]
            [clojure.tools.cli :refer [parse-opts]]
            [clojure.core.async :as async]
            [clojure.string :as s]
            [atomist.time :as time])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn -js->clj+
  "For cases when built-in js->clj doesn't work. Source: https://stackoverflow.com/a/32583549/4839573"
  [x]
  (into {} (for [k (js-keys x)]
             [k (aget x k)])))

(defn env
  "Returns current env vars as a Clojure map."
  []
  (-js->clj+ (.-env js/process)))

(def graphql-endpoint (or (get (env) "GRAPHQL_ENDPOINT") "https://automation.atomist.com/graphql"))

(defn graphql->channel
  "requires an atomist://api-key, and a :team :id in the request

     returns the HTTP response for this team graphql query"
  [o query variables]
  (let [url (gstring/format "%s/team/%s" graphql-endpoint (or (-> o :team :id) (-> o :extensions :team_id)))
        auth-header (gstring/format "Bearer %s" (->> o :secrets (filter #(= "atomist://api-key" (:uri %))) first :value))]
    (if (and url auth-header)
      (client/post
       url
       {:headers {"Authorization" auth-header}
        :body (-> {:query query
                   :variables variables}
                  (json/->str))})
      (throw (ex-info "unable to extract url and token from request" {:request o
                                                                      :query query})))))

(defn head-commits->channel
  "  requires that the team id is already in the request
     returns channel emitting coll of {:keys [analysis repo]}"
  [request type name]
  (go
    (let [response (<! (graphql->channel request graphql/head-commits {:type type :name name}))]
      (if-let [head-commits (-> response
                                :body
                                :data
                                :headCommitsWithFingerprint)]
        head-commits
        (log/warnf "no head-commits for %s %s" type name)))))

(defn github-app-installation-token->channel [request owner]
  (go
    (-> (<! (graphql->channel request graphql/githubAppInstallationByOwner {:name owner}))
        :body
        :data
        :GitHubAppInstallation
        first
        :token
        :secret)))

(defn repo-refs->channel
  [request ref-filter]
  (let [c (chan)]
    (go
      (let [response (<! (graphql->channel request graphql/repos {}))]
        (doseq [org (-> response :body :data :Org)]
          (let [token (or (-> org :scmProvider :credential :secret) (<! (github-app-installation-token->channel request (:owner org))))]
            (doseq [repo (:repos org)
                    :let [ref {:owner (:owner org)
                               :owner-id (:id org)
                               :repo (:name repo)
                               :repo-id (:id repo)
                               :branch "master"}]]
              (when (<! (ref-filter (assoc request :ref ref :token token) ref))
                (>! c (assoc request
                             :token token
                             :ref ref)))))))
      (async/close! c))
    c))

(defn linked-slack-team->channel
  "  requires that the team id is already in the request
     returns channel emitting nil or {:keys [id name]} of linked ChatTeam"
  [request]
  (go
    (let [response (<! (graphql->channel request graphql/chatTeam {:id (-> request :team :id)}))]
      (if-let [chat-team (-> response
                             :body
                             :data
                             :Team
                             first
                             :chatTeams
                             first)]
        chat-team
        (log/warnf "no chatTeams linked to Team %s" (-> request :team :id))))))

(defn linked-person->channel
  "query for all persons linked to a chatUserId, but filter for only Persons representing
     scmIds for a resource provider with provider-id
   channel returns
     nil when there's no linked person
     OR Person map

   if you want to extract a credential from a person use (-> person :scmId :credential :secret)"
  [ch-request chat-id]
  (go
    (let [response (<! (graphql->channel ch-request graphql/linkedScmId {:chatUserId chat-id}))]
      (if-let [persons (-> response
                           :body
                           :data
                           :ChatId)]
        (->> persons
             (filter (fn [person] (= "github_com" (-> person :person :scmId :provider :providerType))))
             first
             :person)
        (log/warnf "no persons for chatUserId %s: %s" chat-id (:status response))))))

(defn github-provider->channel [ch-request]
  (go
    (if-let [providers (-> (<! (graphql->channel ch-request graphql/scmProviders {}))
                           :body
                           :data
                           :SCMProvider)]
      (->> providers
           (filter (fn [provider] (= "github_com" (-> provider :providerType))))
           first))))

(defn linked-repos->channel
  "query for all repos linked to this channel
   channel returns
     nil when there's no linked repo
     OR an array of linked Repos"
  [ch-request channel-id]
  (go
    (if-let [linked-repos (-> (<! (graphql->channel ch-request graphql/linkedRepos {:channelId channel-id}))
                              :body
                              :data
                              :ChatChannel
                              first
                              :repos)]
      linked-repos
      (log/warnf "no linked repos for channelId %s" channel-id))))

(defn repo-query->channel [ch-request repo-name]
  (go
    (let [response (<! (graphql->channel ch-request graphql/repo {:name repo-name}))]
      (-> response
          :body
          :data
          :Repo
          first))))

