(ns atomist.promise
  (:require-macros [cljs.core.async.macros :refer [go]])
  (:require [cljs.core.async :refer [chan <! >! close!]]
            [atomist.cljs-log :as log]))

(defn from-promise
  "turn a promise into a channel
    - channel will emit Promise value and then close
        - truthy Promise values go through js->clj, which works for primitives and js Objects
        - non-truthy Promise values are sent as the keyword :done
        - Promise errors are sent as a map with a {:failure key}"
  ([promise]
   (from-promise
    promise
    (fn [result] (if result (js->clj result :keywordize-keys true) :done))
    (fn [error] {:failure error})))
  ([promise value-handler failure-handler]
   (let [c (chan)
         {:keys [async]} (meta value-handler)]
     (.catch
      (.then promise (fn [result]
                       (go (>! c (if async 
                                   (<! (value-handler result))
                                   (value-handler result)))
                           (close! c))))
      (fn [error]
        (log/error "promise error:  " error)
        (go (>! c (failure-handler error))
            (close! c))))
     c)))

(defn chan->promise
  "convert a channel into a Promise
     - value of Promise can not be nil (Promise reject if chan emits a nil)"
  [chan]
  (js/Promise.
   (fn [accept reject]
     (go
       (try
         (let [v (<! chan)]
           (if v
             (accept v)
             (do
               (log/warn "reject nil on chan->promise")
               (reject (clj->js {:fail "nil value on channel"})))))
         (catch :default t
           (log/error t " js Promise will reject")
           (log/error chan)
           (reject (clj->js {:fail "failure to process chan"
                             :error t}))))))))

(defn chan->obj-promise
  "convert the next value of a channel into a Promise
     - Project will reject if next val is nil
     - Promise values will be #js converted with clj->js"
  [chan]
  (js/Promise.
   (fn [accept reject]
     (go
       (try
         (let [v (<! chan)]
           (if v
             (accept (clj->js v))
             (reject (clj->js {:fail "empty chan value"}))))
         (catch :default t
           (log/error t " js Promise will reject")
           (log/error chan)
           (reject (clj->js {:fail "Promise rejected"
                             :error t}))))))))
