;; Copyright © 2021 Atomist, Inc.
;;
;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at
;;
;;     http://www.apache.org/licenses/LICENSE-2.0
;;
;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns atomist.docker
  (:require [atomist.async :refer-macros [<? go-safe] :refer [map-reduce]]
            [atomist.cljs-log :as log]
            [atomist.json :as json]
            [clojure.data :as data]
            [cljs.core.async :as async :refer [>!]]
            [cljs.pprint :refer [pprint]]
            [goog.crypt.base64 :refer [decodeString]]
            [goog.string :as gstring]
            [http.client :as client]))

(defn- get-image-details-location
  "Weird re-direct thingy"
  [domain access-token repository image-digest]
  (go-safe
   (log/infof "Fetching image-details location for %s/%s (image id %s)" domain repository image-digest)
   (let [image-details (<? (client/get (gstring/format "https://%s/v2/%s/blobs/%s" domain repository image-digest)
                                       {:headers {"Authorization" (gstring/format "Bearer %s" access-token)}}))
         location (-> image-details :headers :location)]
     (if (and location
              (#{307 302 200} (:status image-details)))
       location
       (log/warnf "Could not find image-details location for %s/%s (image id %s)" domain repository image-digest)))))

(defn- get-image-labels
  "Fetch labels given an image-id"
  [domain access-token repository image-id]
  (go-safe
   (log/infof "Fetching image labels for %s/%s@%s" domain repository image-id)
   (if-let [location (<? (get-image-details-location domain access-token repository image-id))]
     (let [metadata (<? (client/get location {}))]
       (if (= 200 (:status metadata))
         (-> metadata :body (str) (json/->obj) :config :Labels)
         (log/warnf "Error fetching labels for %s/%s (image id %s) -> %s" domain repository image-id (:status metadata))))
     (log/warnf "Could not find labels for %s/%s (image id %s)" domain repository image-id))))

(defn- get-manifest
  "Fetch a manifest by digest or tag. Can return a manifest or a manifest-list"
  [domain access-token repository tag-or-digest]
  (go-safe
   (log/infof "Fetching manifest %s/%s:%s" domain repository tag-or-digest)
   (let [response (<? (client/get (gstring/format "https://%s/v2/%s/manifests/%s" domain repository tag-or-digest)
                                  {:headers {"Authorization" (gstring/format "Bearer %s" access-token)
                                             "Accept" "application/vnd.docker.distribution.manifest.v2+json,application/vnd.docker.distribution.manifest.list.v2+json"}}))]
     (if (= 200 (:status response))
       (let [manifest (-> response :body (str) (json/->obj))]
         ;; sometimes we get a list of manifests if the image is multi-platform
         (if (->> response :headers :content-type (= "application/vnd.docker.distribution.manifest.list.v2+json"))
           (log/infof "Found a manifest list for %s/%s:%s" domain repository tag-or-digest)
           (log/infof "Fetched normal manifest %s/%s:%s" domain repository tag-or-digest))
         (assoc manifest :digest (-> response :headers :docker-content-digest)))
       (log/warnf "error fetching manifest for %s/%s:%s status %s" domain repository tag-or-digest (:status response))))))

(defn- resolve-platform-manifest
  "Grab a manifest and associate it with its platform"
  [domain access-token repository platform-manifest]
  (go-safe
   (try
     (let [manifest (<? (get-manifest domain access-token repository (:digest platform-manifest)))]
       (assoc-in manifest [:manifest :platform] (:platform platform-manifest)))
     (catch :default _))))

(defn- get-manifests-for-list
  "Resolve all manifests in a given list"
  [domain access-token repository manifest-list]
  (go-safe
   (log/infof "Resolving all manifests for list %s/%s" domain repository)
   (<? (->>
        manifest-list
        (filter #(= (:mediaType %) "application/vnd.docker.distribution.manifest.v2+json"))
        (map (partial resolve-platform-manifest domain access-token repository))
        (async/merge)
        (async/reduce (fn [acc i]
                        (if (instance? js/Error i)
                          (do
                            (log/errorf i "Failed to resolve a manifest from manifest-list")
                            acc)
                          (conj acc i))) [])))))

(defn get-manifests
  "Get all manifests for a tag or digest. Will contain platform for fat-manifests"
  [domain access-token repository tag-or-digest]
  (go-safe
   (if-let [manifest (<? (get-manifest domain access-token repository tag-or-digest))]
     (if-let [manifest-list (:manifests manifest)]
       (<? (get-manifests-for-list domain access-token repository manifest-list))
       ;; TODO - can we resolve platform via some API?
       [manifest])
     (log/warnf "Could not find manifests for %s/%s:%s" domain repository tag-or-digest))))

(defn get-labelled-manifests
  "Add labels for each image to its manifest"
  [domain access-token repository tag-or-digest]
  (go-safe
   (log/infof "Fetching labelled manifests for %s/%s:%s" domain repository tag-or-digest)
   (if-let [manifests (<? (get-manifests domain access-token repository tag-or-digest))]
     (<? (map-reduce (fn [manifest]
                       (go-safe
                        (if-let [labels (<? (get-image-labels domain access-token repository (-> manifest :config :digest)))]
                          (do
                            (log/debugf "Adding labels to manifest %s..." labels)
                            (assoc manifest :labels labels))
                          (do
                            (log/warnf "Could not find labels for %s/%s:%s" domain repository tag-or-digest)
                            manifest))))
                     manifests))
     (log/warnf "Could not find manifests for %s/%s:%s" domain repository tag-or-digest))))

(defn ->platform
  [platform parent-image]
  (when platform
    [(merge {:schema/entity-type :docker/platform
             :docker.platform/image parent-image
             :schema/entity "$platform"
             :docker.platform/architecture (:architecture platform)
             :docker.platform/os (:os platform)}
            (when-let [variant (:variant platform)]
              {:docker.platform/variant variant}))]))

(defn ->image-layers-entities
  "Generate entities for an image and manifest/label details retrieved from an api"
  [domain repository manifest & [tag]]
  (let [labels (:labels manifest)
        labels-tx (->>
                   (seq labels)
                   (map (fn [[k v]]
                          (let [ref (str "$" (name k))]
                            [ref
                             {:schema/entity-type :docker.image/label
                              :schema/entity ref
                              :docker.image.label/name (name k)
                              :docker.image.label/value (str v)}]))))
        sha (:org.label-schema.vcs-ref labels)]
    (log/infof "label-map:  %s" labels-tx)
    (log/infof "label-map vcs-ref:  %s" sha)
    (concat
     [(merge
       {:schema/entity-type :docker/image
        :schema/entity "$docker-image"
        :docker.image/digest (:digest manifest)
        ;; TODO - these are mutable, so we should be able to add/remove them?
        :docker.image/tags (if tag [tag] [])
        :docker.image/labels (->> labels-tx
                                  (map first)
                                  (into []))
        :docker.image/layers {:set (map #(str "$layer-" (:digest %)) (:layers manifest))}}
       (when sha
         {:docker.image/sha sha})
       (when (and domain repository)
         {:docker.image/repository "$repository"}))]
     (when (and domain repository)
       [{:schema/entity-type :docker/repository
         :schema/entity "$repository"
         :docker.repository/host domain
         :docker.repository/repository repository}])
     (when labels-tx (->> (seq labels-tx)
                          (map second)
                          (into [])))
     (when manifest
       (flatten
        (map-indexed
         (fn [index {:keys [size digest]}]
           [{:schema/entity-type :docker.image/layer
             :schema/entity (str "$layer-" digest)
             :docker.image.layer/ordinal index
             :docker.image.layer/blob (str "$blob-" digest)}
            {:schema/entity-type :docker.image/blob
             :schema/entity (str "$blob-" digest)
             :docker.image.blob/size size
             :docker.image.blob/digest digest}])
         (:layers manifest))))
     (->platform (:platform manifest) "$docker-image"))))

(defn ->nice-image-name
  [docker-image]
  (str (-> docker-image :docker.image/repository :docker.repository/repository) "@" (:docker.image/digest docker-image)))

(defn layers-match?
  "Do layers in manifest match that of parent image?"
  [parent-image manifest]
  (let [base-layer-digests (->> manifest :layers (map :digest))
        current-image-digests (->> parent-image
                                   :docker.image/layers
                                   (sort-by :docker.image.layer/ordinal)
                                   (take (count base-layer-digests))
                                   (map :docker.image.layer/blob)
                                   (map :docker.image.blob/digest))]
    (if (= base-layer-digests current-image-digests)
      true
      (do
        (log/warnf "%s - FROM image layers don't match: %s"
                   (->nice-image-name parent-image)
                   (data/diff current-image-digests base-layer-digests))
        false))))

(defn matching-image
  "Return matching manifest (if any)"
  [parent-image manifests]
  (some->> manifests
           (filter (partial layers-match? parent-image))
           first))